/********************************************************************
 * Copyright (c) 2021 by SK karoly.saly@matrasoft.hu				*
 *																	*
 * s7com32 - s7com32.c												*
 ********************************************************************/
#define WIN32_LEAN_AND_MEAN
#include <skclib32.h>
#include <s7com32.h>
#include "private.h"

static INT32	m_nIFMode	= 0;
static INT32	m_nAreaCode[S7AT_MAX]	= {
	S7ATC_INPUT,	S7ATC_OUTPUT,	S7ATC_FLAG,	S7ATC_DB,	S7ATC_TIMER,	S7ATC_COUNTER
};
static INT32	m_nVarCode[S7VT_MAX]	= {
		S7VTC_BIT,	S7VTC_BYTE,	S7VTC_CHAR,		S7VTC_CHAR,	S7VTC_WORD,	S7VTC_INT,	S7VTC_DWORD,	S7VTC_DINT,
		S7VTC_TIME,	S7VTC_TOD,	S7VTC_S5TIME,	S7VTC_REAL,	S7VTC_DT,	S7VTC_DATE,	S7VTC_TIMER,	S7VTC_COUNTER
};
// **********************************************************************************
INT32 __stdcall pS7_TestInterface(LPS7INTERFACE lpIF) {
LPTIDDATA		pTid;
INT32			nRes;

	pTid = _getptd();
	pTid->nSysErrNo = 0;
	nRes = !lpIF || (lpIF->nStructID != STRUCTID_S7IF) ? S7ERR_INVALIDINTERFACE : S7ERR_NONE;
	pTid->nErrNo = nRes;
	return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7_TestConnection(LPS7CONNECTION lpCon, BOOL bConnected, BOOL bClrAddEn) {
LPTIDDATA		pTid;
LPS7CONNECTION	pAct;
INT32			nRes;

	pTid = _getptd();
	pTid->nSysErrNo = 0;
	nRes = !lpCon || (lpCon->nStructID != STRUCTID_S7CON) ? S7ERR_INVALIDCONNECTION : S7ERR_NONE;
	if (!nRes) nRes = pS7_TestInterface(lpCon->pIF);
	if (!nRes) {
		nRes = S7ERR_INVALIDCONNECTION;
		pAct = lpCon->pIF->pConnection;
		while (lpCon && pAct && nRes) {
			if (pAct->nConID == lpCon->nConID) {
				nRes = S7ERR_NONE;
			}
			else {
				pAct = pAct->pNextCon;
			}
		}
	}
	if (!nRes) {
		if (bClrAddEn) lpCon->bAddEnabled = FALSE;
		if (bConnected && !lpCon->bConnected) nRes = S7ERR_DISCONNECTED;
	}
	pTid->nErrNo = nRes;
	return nRes;
}
// **********************************************************************************
HANDLE __stdcall S7_Interface_Init(UINT32 nMode) {
LPTIDDATA		pTid;
LPS7INTERFACE	pIF;
INT32			nMask, nRes, nErr;
BOOL			bCloseLib;

	pTid = _getptd();
	nRes = nErr = 0;
	pIF = NULL;
	if (nMode > IFM_LAST) {
		nRes = S7ERR_INVALIDMODE;
	}
	else {
		nMask = 1 << nMode;
		if (m_nIFMode & nMask) {
			nRes = S7ERR_INTERFACEINUSE;
		}
		else {
			pIF = (LPS7INTERFACE) GetMem(GMM_CLEAR, sizeof(S7INTERFACE));
			if (pIF) {
				pIF->nStructID = STRUCTID_S7IF;
				pIF->nMode = nMode;
				if (nMode != IFM_TCP) nRes = pS7Online_Init(pIF, &nErr);
			}
			else {
				nRes = S7ERR_NOMEMORY;
			}
		}
	}
	if (!nRes) {
		m_nIFMode |= nMask;
	}
	else if (pIF) {
		if (pIF->nMode != IFM_TCP) {
			bCloseLib = (m_nIFMode & ~(1 << IFM_TCP)) == 0;
			pS7Online_Done(pIF, bCloseLib);
		}
		FreeMem(pIF);
		pIF = NULL;
	}
	pTid->nSysErrNo = nErr;
	pTid->nErrNo = nRes;
	return pIF;	
}
// **********************************************************************************
HANDLE __stdcall S7_Interface_Done(HANDLE hIF) {
LPTIDDATA		pTid;
LPS7INTERFACE	pIF;
INT32			nRes;
BOOL			bCloseLib;

	pTid = _getptd();
	pIF = (LPS7INTERFACE) hIF;
	nRes = pS7_TestInterface(pIF);
	if (!nRes) {
		while (pIF->pConnection) {
			S7_Connection_Done(pIF->pConnection);
		}
		m_nIFMode &= ~(1 << pIF->nMode);
		if (pIF->nMode != IFM_TCP) {
			bCloseLib = (m_nIFMode & ~(1 << IFM_TCP)) == 0;
			pS7Online_Done(pIF, bCloseLib);
		}
		FreeMem(pIF);
	}
	pTid->nSysErrNo = 0;
	pTid->nErrNo = nRes;
	return NULL;
}
// **********************************************************************************
HANDLE __stdcall S7_Connection_Init(HANDLE hIF, LPSTR lpName, INT32 nTimeOut, INT32 nAddr, INT32 nRack, INT32 nSlot, INT32 nDefPduSize) {
LPTIDDATA		pTid;
LPS7INTERFACE	pIF;
LPS7CONNECTION	pCon;
INT32			nRes;

	pTid = _getptd();
	pIF = (LPS7INTERFACE) hIF;
	nRes = pS7_TestInterface(pIF);
	if (nRes) return NULL;
	pCon = (LPS7CONNECTION) GetMem(GMM_CLEAR, sizeof(S7CONNECTION));
	if (pCon) {
		pIF->nNextID++;
		pCon->nConID = pIF->nNextID;
		pIF->pConnection = Chain_AddItem(pCon, pIF->pConnection);
		pCon->pIF = pIF;
		pCon->nStructID = STRUCTID_S7CON;
		pCon->nTimeOut = nTimeOut >= 1000 ? nTimeOut : 1000;
		pCon->nAddr = nAddr;
		pCon->nRack = nRack;
		pCon->nSlot = nSlot;
		pCon->nPDUSize = MAX_S7PDUSIZE;
		StrCpyN(pCon->cName, lpName, S7MAX_NAMESIZE);
		nRes = S7_Connect(pCon);
		if (!pCon->nPDUSize) pCon->nPDUSize = nDefPduSize;
	}
	else {
		nRes = S7ERR_NOMEMORY;
		pTid->nErrNo = nRes;
	}
    return pCon;	
}
// **********************************************************************************
HANDLE __stdcall S7_Connection_Done(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, TRUE);
	if (!nRes) {
		nRes = S7_Disconnect(hCon);
		pCon->pIF->pConnection = Chain_DelItem(pCon->nConID, pCon->pIF->pConnection);
		FreeMem(hCon);
	}
	return NULL;
}
// **********************************************************************************
BOOL __stdcall S7_IsConnected(HANDLE hCon) {
LPS7CONNECTION	pCon;
BOOL			bRet;
INT32			nRes;

	bRet = FALSE;
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	if (!nRes) {
		bRet = pCon->bConnected;
	}
	return bRet;
}
// **********************************************************************************
INT32 __stdcall S7_Connect(HANDLE hCon) {
LPTIDDATA		pTid;
LPS7CONNECTION	pCon;
INT32			nRes, nErr;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, TRUE);
	if (!nRes && !pCon->bConnected) {
		nErr = 0;
		if (pCon->pIF->nMode == IFM_TCP) {
			nRes = pS7IsoTcp_Connect(pCon, &nErr);
		}
		else {
			nRes = pS7Online_Connect(pCon, &nErr);
		}
		if (nRes == S7ERR_NONE) {
			pCon->bConnected = TRUE;
		}
		else {
			S7_Disconnect(hCon);
		}
		pTid->nSysErrNo = nErr;
		pTid->nErrNo = nRes;
	}
	return nRes;	
}
// **********************************************************************************
INT32 __stdcall S7_Disconnect(HANDLE hCon) {
LPTIDDATA		pTid;
LPS7CONNECTION	pCon;
INT32			nRes;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, TRUE);
	if (!nRes && pCon->bConnected) {
		if (pCon->pIF->nMode == IFM_TCP) {
			nRes = pS7IsoTcp_Disconnect(pCon, &pTid->nSysErrNo);
		}
		else {
			nRes = pS7Online_Disconnect(pCon, &pTid->nSysErrNo);
		}
		pTid->nErrNo = nRes;
		FreeMem(pCon->pSndBuf);
		pCon->pSndBuf = pCon->pRcvBuf = NULL;
		MemClr(&pCon->SndPdu, sizeof(S7PDU));
		MemClr(&pCon->RcvPdu, sizeof(S7PDU));
		MemClr(&pCon->RdInfo, sizeof(pCon->RdInfo));
		MemClr(&pCon->nRWErrInfo, sizeof(pCon->nRWErrInfo));
		pCon->nRWErrCnt = 0;
		pCon->nTPDUSize = 0;
		pCon->bConnected = FALSE;
	}
	return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7_Exchange(LPS7CONNECTION lpCon) {
LPTIDDATA		pTid;
INT32			nRes, nErr;

	pTid = _getptd();
	nRes = pS7_TestConnection(lpCon, TRUE, TRUE);
	if (nRes) return nRes;
	if (lpCon->pIF->nMode == IFM_TCP) {
		nRes = pS7IsoTcp_Exchange(lpCon, &nErr);
	}
	else {
		nRes = pS7Online_Exchange(lpCon, &nErr);
	}
	if (!nRes) nRes = pS7PDU_RcvInit(lpCon, &nErr);
	pTid->nSysErrNo = nErr;
	pTid->nErrNo = nRes;
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_AddItem(HANDLE hCon, UINT32 nVT, UINT32 nArea, INT32 nDB, UINT32 nAddr, UINT32 nBit, UINT32 nCount, BOOL bConvert, LPVOID lpData) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
LPS7PDUHEADER	pHdr;
S7ANYPARPTR		pPar;
S7ANYDATAPTR	pDat;
LPSTR			pSrc, pDst;
ITEMINFO		Item;
DATAHEADER		DHdr;
UINT32			i, j;
INT32			nFullAddr, nReqType, nReqCount, nVarSize, nDatSize, nRequestLen, nReplyLen, nParLen, nDatLen, nLength, nVarIdx, nRes, nVTC;
BOOL			bWrite, bCopy;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	if (nRes) return nRes;
	if (!pCon->bAddEnabled) {
		nRes = S7ERR_INVALIDREQUEST;
	}
	else {
		pHdr = (LPS7PDUHEADER) pCon->SndPdu.pData;
		nRequestLen = pCon->SndPdu.nHdrLen;
		nParLen = pCon->SndPdu.nParLen;
		nDatLen = pCon->SndPdu.nDatLen;
		pPar.pStr = pCon->SndPdu.pData + nRequestLen;
		pDat.pStr = pPar.pStr + nParLen;
		nRequestLen += nParLen + nDatLen;
		nReqType = nVTC = m_nVarCode[nVT];
		nReqCount = nCount;
		if (pPar.pReqRW->Hdr.nCount == MAX_S7VARS) {
			nRes = S7ERR_TOOMUCHITEM;
		}
		else if (nVT >= S7VT_MAX) {
			nRes = S7ERR_INVALIDTYPE;
		}
		else if (nArea >= S7AT_MAX) {
			nRes = S7ERR_INVALIDAREA;
		}
		else if ((nVTC != S7VTC_BIT) && nBit) {
			nRes = S7ERR_INVALIDADDRESS;
		}
		else if (!nCount) {
			nRes = S7ERR_INVALIDCOUNT;
		}
		else {
			switch (nVTC) {
			case S7VTC_BIT:
				if (nBit > 7) {
					nRes = S7ERR_INVALIDADDRESS;
				}
				else if ((nCount > 1) && ((nCount & 7) || nBit)){
					nRes = S7ERR_INVALIDCOUNT;
				}
				if ((nCount > 1) && !nRes) nReqType = S7VTC_CHAR;
			case S7VTC_BYTE:
			case S7VTC_CHAR:
				nVarSize = sizeof(INT8);
				break;
			case S7VTC_DATE:
			case S7VTC_S5TIME:
				nReqType = S7VTC_CHAR;
			case S7VTC_WORD:
			case S7VTC_INT:
			case S7VTC_COUNTER:
			case S7VTC_TIMER:
				nVarSize = sizeof(INT16);
				break;
			case S7VTC_TOD:
			case S7VTC_TIME:
				nReqType = S7VTC_CHAR;
			case S7VTC_DWORD:
			case S7VTC_DINT:
			case S7VTC_REAL:
				nVarSize = sizeof(INT32);
				break;
			case S7VTC_DT:
				nReqType = S7VTC_CHAR;
				nVarSize = sizeof(S7DATETIME);
				break;
			default:
				nRes = S7ERR_INVALIDTYPE;
			}
		}
	}
	if (!nRes) {
		nDatSize = nVarSize * nCount;
		if (nVTC != nReqType) {
			if (nVTC == S7VTC_BIT) nDatSize >>= 3;
			nReqCount = nDatSize;
		}
		MemClr(&Item, sizeof(ITEMINFO));
		Item.nHead[0] = 0x12;
		Item.nHead[1] = 0x0a;
		Item.nHead[2] = 0x10;
		Item.nVT = nReqType;
		Item.nCount = ByteSwap16(nReqCount);
		Item.nDB = ByteSwap16(nDB);
		nFullAddr = (nVTC == S7VTC_COUNTER) || (nVTC == S7VTC_TIMER) ? nAddr : (nAddr << 3) + nBit;
		Item.nAddress = ByteSwap32(nFullAddr);
		Item.nArea = m_nAreaCode[nArea];
		bWrite = pPar.pReqRW->Hdr.nFunc == PDUFN_WRITE;
		nRequestLen += sizeof(ITEMINFO);
		nReplyLen = pCon->nReplyLen;
		if (bWrite) {
			if (nDatLen & 1) {
				pDat.pStr[nDatLen] = 0;
				nDatLen++;
				nRequestLen++;
			}
			nRequestLen += nDatSize + sizeof(DATAHEADER);
			nReplyLen++;
		}
		else {
			if (nReplyLen & 1) nReplyLen++;
			nReplyLen += nDatSize + sizeof(DATAHEADER);
		}
		if ((nRequestLen > pCon->nPDUSize) || (nReplyLen > pCon->nPDUSize)) nRes = S7ERR_TOOBIGSIZE;
	}
	if (!nRes) {
		nVarIdx = pPar.pReqRW->Hdr.nCount++;
		if (bWrite) {
			nLength = nDatSize;
			DHdr.nRetVal = 0;
			switch (nReqType) {
			case S7VTC_BIT:
				DHdr.nTS = TS_BIT;
				break;
			case S7VTC_INT:
			case S7VTC_DINT:
				DHdr.nTS = TS_INT;
				break;
			case S7VTC_REAL:
				DHdr.nTS = TS_REAL;
				break;
			case S7VTC_BYTE:
			case S7VTC_WORD:
			case S7VTC_DWORD:
				DHdr.nTS = TS_BYTE;
				break;
			default:
				DHdr.nTS = TS_OCTET;
			}
	// TS_BIT, TS_REAL, TS_OCTET : nLength = nDatSize
	// TS_BYTE, TS_INT : nLength = nDatSize * 8
			if ((DHdr.nTS == TS_BYTE) || (DHdr.nTS == TS_INT)) nLength <<= 3;
			DHdr.nLength = ByteSwap16(nLength);
			pDst = pDat.pStr + sizeof(ITEMINFO);
			if (nDatLen) MemMove(pDst, pDat.pStr, nDatLen);
			pDst += nDatLen;
			MemCpy(pDst, &DHdr, sizeof(DATAHEADER));
			pDst += sizeof(DATAHEADER);
			bCopy = bConvert ? ((nVT == S7VT_CHAR) || (nVT == S7VT_DT) ? TRUE : FALSE) : TRUE;
			if (bCopy) {
				MemCpy(pDst, lpData, nDatSize);
			}
			else {
				pSrc = lpData;
				for (i = 0; i < nCount; i++) {
					switch (nVT) {
					case S7VT_BIT:
						if (nCount == 1) {
							*pDst = *((LPBOOL) pSrc) ? 1 : 0;
						}
						else {
							for (j = 0; j < 8; j++) {
								SetBit(pDst, j, *((LPBOOL) pSrc) ? 1 : 0);
								if (j < 7) {
									pSrc += sizeof(INT32);
									i++;
								}
							}
						}
						break;
					case S7VT_S5TIME:
					case S7VT_TIMER:
						S7_SetS5Time(pDst, *((LPUINT32) pSrc));
						break;
					case S7VT_COUNTER:
						S7_SetCounter(pDst, *((LPUINT16) pSrc));
						break;
					case S7VT_BYTE:
					case S7VT_SHORT:
						*((LPUINT8) pDst) = *((LPUINT8) pSrc);
						break;
					case S7VT_DATE:
					case S7VT_WORD:
					case S7VT_INT:
						*((LPUINT16) pDst) = ByteSwap16(*((LPUINT16) pSrc));
						break;
					case S7VT_DWORD:
					case S7VT_DINT:
					case S7VT_TIME:
					case S7VT_TOD:
					case S7VT_REAL:
						*((LPUINT32) pDst) = ByteSwap32(*((LPUINT32) pSrc));
						break;
					}
					pSrc += sizeof(INT32);
					pDst += nVarSize;
				}
			}
			nDatLen += nDatSize + sizeof(DATAHEADER);
		}
		else {
			pCon->RdInfo[nVarIdx].nVT = nVT;
			pCon->RdInfo[nVarIdx].nCount = nCount;
			pCon->RdInfo[nVarIdx].nSize = nVarSize;
			pCon->RdInfo[nVarIdx].nCvt = bConvert ? 1 : 0;
			pCon->RdInfo[nVarIdx].pData.pVoid = lpData;
		}
		MemCpy(pDat.pStr, &Item, sizeof(ITEMINFO));
		nParLen += sizeof(ITEMINFO);
		pCon->SndPdu.nParLen = nParLen;
		pHdr->nParLen = ByteSwap16(nParLen);
		pCon->SndPdu.nDatLen = nDatLen;
		pHdr->nDatLen = ByteSwap16(nDatLen);
		pCon->SndPdu.nTotLen = nRequestLen;
		pCon->nReplyLen = nReplyLen;
	}
	pTid->nErrNo = nRes;
	return nRes;
}    
// **********************************************************************************
INT32 __stdcall S7_GetItemMaxCount(HANDLE hCon, BOOL bNew, UINT32 nVT, LPUINT32 lpCount) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
LPS7PDUHEADER	pHdr;
S7ANYPARPTR		pPar;
UINT32			nCount, nDelta, nVarSize;
INT32			nRequestLen, nReplyLen, nRes;
BOOL			bWrite;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	if (lpCount) *lpCount = 0;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	if (nRes) return nRes;
	pHdr = (LPS7PDUHEADER) pCon->SndPdu.pData;
	pPar.pStr = pCon->SndPdu.pData + pCon->SndPdu.nHdrLen;
	if (!pCon->bAddEnabled) {
		nRes = S7ERR_INVALIDREQUEST;
	}
	else if (pPar.pReqRW->Hdr.nCount == MAX_S7VARS) {
		nRes = S7ERR_TOOMUCHITEM;
	}
	else {
		switch (nVT) {
		case S7VT_BIT:
		case S7VT_BYTE:
		case S7VT_SHORT:
		case S7VT_CHAR:
			nVarSize = sizeof(INT8);
			break;
		case S7VT_DATE:
		case S7VT_S5TIME:
		case S7VT_WORD:
		case S7VT_INT:
		case S7VT_COUNTER:
		case S7VT_TIMER:
			nVarSize = sizeof(INT16);
			break;
		case S7VT_TOD:
		case S7VT_TIME:
		case S7VT_DWORD:
		case S7VT_DINT:
		case S7VT_REAL:
			nVarSize = sizeof(INT32);
			break;
		case S7VT_DT:
			nVarSize = sizeof(S7DATETIME);
			break;
		default:
			nRes = S7ERR_INVALIDTYPE;
		}
	}
	if (!nRes) {
		bWrite = pPar.pReqRW->Hdr.nFunc == PDUFN_WRITE;
		nRequestLen = pCon->SndPdu.nTotLen;
		nReplyLen = pCon->nReplyLen;
		if (bNew) {
			nRequestLen += sizeof(ITEMINFO);
			if (bWrite) {
				nRequestLen += sizeof(DATAHEADER);
			}
			else {
				nReplyLen += sizeof(DATAHEADER);
			}
		}
		if ((nRequestLen >= pCon->nPDUSize) || (nReplyLen >= pCon->nPDUSize)) {
			nCount = 0;
		}
		else {
			if (bWrite) {
				nDelta = pCon->nPDUSize - nRequestLen;
			}
			else {
				nDelta = pCon->nPDUSize - nReplyLen;
			}
			nCount = nDelta / nVarSize;
		}
		if (lpCount) *lpCount = nCount;
	}
	pTid->nErrNo = nRes;
	return nRes;
}    
// **********************************************************************************
static INT32 __stdcall pS7_GetItems(LPS7CONNECTION lpCon, INT32 nLastErr, LPINT32 lpSysErr) {
S7ANYPARPTR		pRcvPar;
S7ANYPARPTR		pSndPar;
S7ANYDATAPTR	pDat;
LPSTR			pDst, pDatStrt;
UINT32			i, j, k, nVT, nSize, nCount;
INT32			nRes, nLength, nTest, nBits;
BOOL			bCopy;

	pRcvPar.pStr = lpCon->RcvPdu.pData + lpCon->RcvPdu.nHdrLen;
	pSndPar.pStr = lpCon->SndPdu.pData + lpCon->SndPdu.nHdrLen;
	pDatStrt = pDat.pStr = pRcvPar.pStr + lpCon->RcvPdu.nParLen;
	if (nLastErr && (nLastErr != S7ERR_BADREPLYSIZE)) return nLastErr;
	nBits = 0;
	nRes = S7ERR_NONE;
	if ((pRcvPar.pReqRW->Hdr.nFunc != PDUFN_READ) || (pRcvPar.pReqRW->Hdr.nCount != pSndPar.pReqRW->Hdr.nCount)) {
		nRes = S7ERR_INVALIDRESPONSE;
	}
	else {
		for (i = 0; i < pRcvPar.pReqRW->Hdr.nCount; i++) {
			lpCon->nRWErrInfo[i] = pDat.pDHdr->nRetVal;
			nLength = ByteSwap16(pDat.pDHdr->nLength);
			if ((pDat.pDHdr->nTS == TS_BYTE) || (pDat.pDHdr->nTS == TS_INT)) nLength >>= 3;
				pDat.pStr += sizeof(DATAHEADER);
			if (lpCon->nRWErrInfo[i] != IORESULT_OK) {
				if (!nRes) nRes = S7ERR_DATERROR;
				pDat.pStr += nLength;
				SetBit(&nBits, i, 1);
			}
			else {
				nVT = lpCon->RdInfo[i].nVT;
				nSize = lpCon->RdInfo[i].nSize;
				nCount = lpCon->RdInfo[i].nCount;
				pDst = lpCon->RdInfo[i].pData.pVoid;
				nTest = nSize * nCount;
				if ((nVT == S7VT_BIT) && (nCount > 1)) nTest >>= 3;
				if (nLength != nTest) {
					if (!nRes) nRes = S7ERR_BADDATSIZE;
					pDat.pStr += nLength;
					SetBit(&nBits, i, 1);
				}
				else {
					bCopy = lpCon->RdInfo[i].nCvt ? ((nVT == S7VT_CHAR) || (nVT == S7VT_DT) ? TRUE : FALSE) : TRUE;
					if (bCopy) {
						MemCpy(pDst, pDat.pVoid, nLength);
						pDat.pStr += nLength;
					}
					else {
						for (j = 0; j < nCount; j++) {
							switch (nVT) {
							case S7VT_BIT:
								if (nCount == 1) {
									*((LPBOOL) pDst) = *pDat.pStr ? TRUE : FALSE;
								}
								else {
									for (k = 0; k < 8; k++) {
										*((LPBOOL) pDst) = GetBit(pDat.pStr, k) ? 1 : 0;
										if (k < 7) {
											pDst += sizeof(INT32);
											j++;
										}
									}
								}
								break;
							case S7VT_S5TIME:
							case S7VT_TIMER:
								*((LPUINT32) pDst) = S7_GetS5Time(pDat.pVoid);
								break;
							case S7VT_COUNTER:
								*((LPUINT32) pDst) = S7_GetCounter(pDat.pVoid);
								break;
							case S7VT_BYTE:
								*((LPUINT32) pDst) = S7_GetByte(pDat.pVoid);
								break;
							case S7VT_SHORT:
								*((LPINT32) pDst) = S7_GetShort(pDat.pVoid);
								break;
							case S7VT_WORD:
							case S7VT_DATE:
								*((LPUINT32) pDst) = S7_GetWord(pDat.pVoid);
								break;
							case S7VT_INT:
								*((LPINT32) pDst) = S7_GetInt(pDat.pVoid);
								break;
							case S7VT_DWORD:
							case S7VT_DINT:
							case S7VT_TIME:
							case S7VT_TOD:
							case S7VT_REAL:
								*((LPUINT32) pDst) = ByteSwap32(*((LPUINT32) pDat.pVoid));
								break;
							}
							pDat.pStr += nSize;
							pDst += sizeof(INT32);
						}
					}
				}
			}
			if ((pDat.pStr - pDatStrt) & 1) pDat.pStr++;
		}
		lpCon->nRWErrCnt = pRcvPar.pReqRW->Hdr.nCount;
	}
	if (!nLastErr) nLastErr = nRes;
	*lpSysErr = nBits;
	return nLastErr;
}    
// **********************************************************************************
static INT32 __stdcall pS7_WriteResult(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
S7ANYPARPTR		pRcvPar;
S7ANYPARPTR		pSndPar;
LPUINT8			pDat;
INT32			i, nRes, nBits;

	pRcvPar.pStr = lpCon->RcvPdu.pData + lpCon->RcvPdu.nHdrLen;
	pSndPar.pStr = lpCon->SndPdu.pData + lpCon->SndPdu.nHdrLen;
	pDat = pRcvPar.pStr + lpCon->RcvPdu.nParLen;
	if ((pRcvPar.pReqRW->Hdr.nFunc != PDUFN_WRITE) || (pRcvPar.pReqRW->Hdr.nCount != pSndPar.pReqRW->Hdr.nCount)) return S7ERR_INVALIDRESPONSE;
	nRes = S7ERR_NONE;
	nBits = 0;
	for (i = 0; i < pRcvPar.pReqRW->Hdr.nCount; i++) {
		lpCon->nRWErrInfo[i] = *pDat;
		if (lpCon->nRWErrInfo[i] != IORESULT_OK) {
			nRes = S7ERR_DATERROR;
			SetBit(&nBits, i, 1);
		}
		pDat++;
	}
	lpCon->nRWErrCnt = pRcvPar.pReqRW->Hdr.nCount;
	*lpSysErr = nBits;
	return nRes;
}    
// **********************************************************************************
INT32 __stdcall S7_SimpleRead(HANDLE hCon, INT32 nVT, INT32 nArea, INT32 nDB, INT32 nAddr, INT32 nBit, INT32 nCount, BOOL bConvert, LPVOID lpData) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
INT32			nRes;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_REQUEST, PDUFN_READ, 0, FALSE);
	if (nRes) return nRes;
	nRes = S7_AddItem(hCon, nVT, nArea, nDB, nAddr, nBit, nCount, bConvert, lpData);
	if (nRes) return nRes;
	nRes = pS7_Exchange(pCon);
	nRes = pTid->nErrNo = pS7_GetItems(pCon, nRes, &pTid->nSysErrNo);
	return nRes; 
}	
// **********************************************************************************
INT32 __stdcall S7_MultiRead_Init(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_REQUEST, PDUFN_READ, 0, FALSE);
	return nRes; 
}	
// **********************************************************************************
INT32 __stdcall S7_MultiRead_Done(HANDLE hCon) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
S7ANYPARPTR		pPar;
INT32			nRes;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	pPar.pStr = pCon->SndPdu.pData + pCon->SndPdu.nHdrLen;
	if (pPar.pReqRW->Hdr.nCount == 0) return S7ERR_NOITEM;
	nRes = pS7_Exchange(pCon);
	nRes = pTid->nErrNo = pS7_GetItems(pCon, nRes, &pTid->nSysErrNo);
	return nRes; 
}	
// **********************************************************************************
INT32 __stdcall S7_SimpleWrite(HANDLE hCon, INT32 nVT, INT32 nArea, INT32 nDB, INT32 nAddr, INT32 nBit, INT32 nCount, BOOL bConvert, LPVOID lpData) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
INT32			nRes;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_REQUEST, PDUFN_WRITE, 0, FALSE);
	if (nRes) return nRes;
	nRes = S7_AddItem(hCon, nVT, nArea, nDB, nAddr, nBit, nCount, bConvert, lpData);
	if (nRes) return nRes;
	nRes = pS7_Exchange(pCon);
	if (nRes) return nRes;
	nRes = pTid->nErrNo = pS7_WriteResult(pCon, &pTid->nSysErrNo);
	return nRes; 
}	
// **********************************************************************************
INT32 __stdcall S7_MultiWrite_Init(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_REQUEST, PDUFN_WRITE, 0, FALSE);
	return nRes; 
}	
// **********************************************************************************
INT32 __stdcall S7_MultiWrite_Done(HANDLE hCon) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
S7ANYPARPTR		pPar;
INT32			nRes;

	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	pPar.pStr = pCon->SndPdu.pData + pCon->SndPdu.nHdrLen;
	if (pPar.pReqRW->Hdr.nCount == 0) return S7ERR_NOITEM;
	nRes = pS7_Exchange(pCon);
	if (nRes) return nRes;
	nRes = pTid->nErrNo = pS7_WriteResult(pCon, &pTid->nSysErrNo);
	return nRes; 
}	
// **********************************************************************************
INT32 __stdcall S7_GetCpuClock(HANDLE hCon, LPSYSTEMTIME lpSt) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
S7ANYDATAPTR	pDat;
INT32			nRes;
BOOL			bErr;
UINT8			nData[4];

// Response: 10-12-14
	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_USERDATA, PDUSFN_GETCLOCK, TG_CLOCK, TRUE);
	if (nRes) return nRes;
	MemClr(nData, sizeof(nData));
	nData[0] = 10;
	pS7PDU_AddData(pCon, nData, sizeof(nData));
	nRes = pS7_Exchange(pCon);
	if (nRes) return nRes;
	pDat.pStr = pCon->RcvPdu.pData + pCon->RcvPdu.nHdrLen + pCon->RcvPdu.nParLen;
	bErr = S7_GetDT(lpSt, &pDat.pClk->Time);
	if (bErr) nRes = S7ERR_INVALIDDATE;
	pTid->nErrNo = nRes;
	return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_SetCpuClock(HANDLE hCon, LPSYSTEMTIME lpSt) {
LPS7CONNECTION	pCon;
LPTIDDATA		pTid;
CLOCKDATA		Clk;
INT32			nRes;
BOOL			bErr;

// Response: 10-12-4
	pTid = _getptd();
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_USERDATA, PDUSFN_SETCLOCK, TG_CLOCK, TRUE);
	if (nRes) return nRes;
	MemClr(&Clk, sizeof(CLOCKDATA));
	Clk.Hdr.nRetVal = 0xFF;
	Clk.Hdr.nTS = TS_OCTET;
	Clk.Hdr.nLength = ByteSwap16(sizeof(CLOCKDATA) - sizeof(DATAHEADER));
//	Clk.nReserved1 = 0;
	Clk.nReserved2 = 0x19;
	bErr = S7_SetDT(&Clk.Time, lpSt);
	if (bErr) {
		nRes = pTid->nErrNo = S7ERR_INVALIDDATE;
	}
	else {
		pS7PDU_AddData(pCon, &Clk, sizeof(CLOCKDATA));
		nRes = pS7_Exchange(pCon);
	}
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_SessionLogin(HANDLE hCon, LPSTR lpPwd) {
LPS7CONNECTION	pCon;
PASSWORDDATA	Pwd;
INT32			nRes, nLen, i;

// Response: 10-12-4
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_USERDATA, PDUSFN_ENTERPWD, TG_SECURITY, TRUE);
	if (nRes) return nRes;
	Pwd.Hdr.nRetVal = 0xFF;
	Pwd.Hdr.nTS = TS_OCTET;
	Pwd.Hdr.nLength = ByteSwap16(sizeof(PASSWORDDATA) - sizeof(DATAHEADER));
	MemFill8(Pwd.cPwd, ' ', sizeof(Pwd.cPwd));
	nLen = StrLen(lpPwd);
	if (nLen > sizeof(Pwd.cPwd)) nLen = sizeof(Pwd.cPwd);
	MemCpy(Pwd.cPwd, lpPwd, nLen);
	for (i = 0; i < sizeof(Pwd.cPwd); i++) {
		Pwd.cPwd[i] ^= 0x55;
		if (i >= 2) Pwd.cPwd[i] ^= Pwd.cPwd[i - 2];
	}
	pS7PDU_AddData(pCon, &Pwd, sizeof(PASSWORDDATA));
	nRes = pS7_Exchange(pCon);
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_SessionLogout(HANDLE hCon) {
LPS7CONNECTION	pCon;
DATAHEADER		Hdr;
INT32			nRes;

// Response: 10-12-4
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_USERDATA, PDUSFN_CANCELPWD, TG_SECURITY, TRUE);
	if (nRes) return nRes;
	MemClr(&Hdr, sizeof(DATAHEADER));
	Hdr.nRetVal = 10;
	pS7PDU_AddData(pCon, &Hdr, sizeof(DATAHEADER));
	nRes = pS7_Exchange(pCon);
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_GetCpuState(HANDLE hCon, LPUINT32 lpState) {
#define SZL_GETCPUSTATE		0x0424
#define S7_STATEUNKNOWN		0x00
#define S7_STATESTOP		0x04
#define S7_STATERUN			0x08
LPS7CONNECTION	pCon;
REQSZLDATA		ReqSZL;
S7ANYDATAPTR	pDat;
INT32			nRes, nState;

// Response: 10-12-32
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_USERDATA, PDUSFN_READSZL, TG_SZL, TRUE);
	if (nRes) return nRes;
	MemClr(&ReqSZL, sizeof(REQSZLDATA));
	ReqSZL.Hdr.nRetVal = 0xFF;
	ReqSZL.Hdr.nTS = TS_OCTET;
	ReqSZL.Hdr.nLength = ByteSwap16(sizeof(REQSZLDATA) - sizeof(DATAHEADER));
	ReqSZL.nID = ByteSwap16(SZL_GETCPUSTATE);
//	ReqSZL.nIndex = 0;
	pS7PDU_AddData(pCon, &ReqSZL, sizeof(REQSZLDATA));
	nRes = pS7_Exchange(pCon);
	pDat.pStr = pCon->RcvPdu.pData + pCon->RcvPdu.nHdrLen + pCon->RcvPdu.nParLen;
	if (!nRes) {
		switch (pDat.pState->nState) {
		case S7_STATEUNKNOWN:
			nState = S7_CPUUNKNOWN;
			break;
		case S7_STATERUN:
			nState = S7_CPURUN;
			break;
		default:
			nState = S7_CPUSTOP;
		}
	}
	else {
		nState = S7_STATEUNKNOWN;
	}
	if (lpState) *lpState = nState;
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_CpuStart(HANDLE hCon, BOOL bCold) {
LPS7CONNECTION	pCon;
INT32			nRes;

// Response: 12-1
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_REQUEST, PDUFN_CPUSTART, bCold, TRUE);
	if (nRes) return nRes;
	nRes = pS7_Exchange(pCon);
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_CpuStop(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes;

// Response: 12-1
	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7PDU_SndInit(pCon, PDUT_REQUEST, PDUFN_CPUSTOP, 0, TRUE);
	if (nRes) return nRes;
	nRes = pS7_Exchange(pCon);
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_GetLastError(LPINT32 lpSysErr) {
LPTIDDATA		pTid;

	pTid = _getptd();
	if (lpSysErr) *lpSysErr = pTid->nSysErrNo;
    return pTid->nErrNo;
}
// **********************************************************************************
INT32 __stdcall S7_IsReadWriteError(HANDLE hCon) {
LPS7CONNECTION		pCon;
INT32				nRes;

	pCon = (LPS7CONNECTION) hCon;
	nRes = !pCon || (pCon->nStructID != STRUCTID_S7CON) ? 0 : pCon->nRWErrCnt;
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_GetReadWriteError(HANDLE hCon, INT32 nIndex, LPINT32 lpIOErr) {
LPS7CONNECTION		pCon;
INT32				nRes, nErr;

	nErr = 0;
	pCon = (LPS7CONNECTION) hCon;
	if (!pCon || (pCon->nStructID != STRUCTID_S7CON)) {
		nRes = S7ERR_INVALIDCONNECTION;
	}
	else if (nIndex >= pCon->nRWErrCnt) {
		nRes = S7ERR_INVALIDREQUEST;
	}
	else {
		nErr = pCon->nRWErrInfo[nIndex];
		nRes = 0;
	}
	if (lpIOErr) *lpIOErr = nErr;
    return nRes;
}
// **********************************************************************************
static INT32 __stdcall pTestStruct(HANDLE hAny) {
INT32		nRes;

	nRes = hAny && (((LPS7COMMONIFCON) hAny)->nStructID == STRUCTID_S7CON) ? pS7_TestConnection((LPS7CONNECTION) hAny, FALSE, FALSE) : pS7_TestInterface((LPS7INTERFACE) hAny);
	return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_GetUserData(HANDLE hAny) {
LPS7COMMONIFCON	pCmn;
INT32			nRes, nRet;

	pCmn = (LPS7COMMONIFCON) hAny;
	nRes = pTestStruct(hAny);
	nRet = nRes ? 0 : pCmn->nUser;
	return nRet;
}
// **********************************************************************************
INT32 __stdcall S7_SetUserData(HANDLE hAny, INT32 nData) {
LPS7COMMONIFCON	pCmn;
INT32			nRes, nRet;

	pCmn = (LPS7COMMONIFCON) hAny;
	nRes = pTestStruct(hAny);
	if (nRes) {
		nRet = 0;
	}
	else {
		nRet = pCmn->nUser;
		pCmn->nUser = nData;
	}
	return nRet;
}
// **********************************************************************************
LPVOID __stdcall S7_GetUserPtr(HANDLE hAny) {
LPS7COMMONIFCON	pCmn;
INT32			nRes;
LPVOID			pRet;

	pCmn = (LPS7COMMONIFCON) hAny;
	nRes = pTestStruct(hAny);
	pRet = nRes ? NULL : pCmn->pUser;
	return pRet;
}
// **********************************************************************************
LPVOID __stdcall S7_SetUserPtr(HANDLE hAny, LPVOID lpPtr) {
LPS7COMMONIFCON	pCmn;
INT32			nRes;
LPVOID			pRet;

	pCmn = (LPS7COMMONIFCON) hAny;
	nRes = pTestStruct(hAny);
	if (nRes) {
		pRet = NULL;
	}
	else {
		pRet = pCmn->pUser;
		pCmn->pUser = lpPtr;
	}
	return pRet;
}
// **********************************************************************************
INT32 __stdcall S7_GetConName(HANDLE hCon, LPSTR lpName) {
LPS7CONNECTION	pCon;
INT32			nRes, nRet;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	if (nRes) {
		if (lpName) lpName[0] = 0;
		nRet = 0;
	}
	else {
		nRet = StrCpy(lpName, pCon->cName);
	}
	return nRet;
}
// **********************************************************************************
void __stdcall S7_SetConName(HANDLE hCon, LPSTR lpName) {
LPS7CONNECTION	pCon;
INT32			nRes;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	if (!nRes) {
		StrCpyN(pCon->cName, lpName, S7MAX_NAMESIZE);
	}
}
// **********************************************************************************
INT32 __stdcall S7_GetMaxPduSize(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes, nRet;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	nRet = nRes ? 0 : pCon->nPDUSize;
	return nRet;
}
// **********************************************************************************
INT32 __stdcall S7_GetRequestPduSize(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes, nRet;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	nRet = nRes ? 0 : pCon->SndPdu.nTotLen;
	return nRet;
}
// **********************************************************************************
INT32 __stdcall S7_GetReplyPduSize(HANDLE hCon) {
LPS7CONNECTION	pCon;
INT32			nRes, nRet;

	pCon = (LPS7CONNECTION) hCon;
	nRes = pS7_TestConnection(pCon, FALSE, FALSE);
	nRet = nRes ? 0 : pCon->nReplyLen;
	return nRet;
}
// **********************************************************************************
INT32 __stdcall S7_GetInterfaceMode(HANDLE hIF) {
LPS7INTERFACE	pIF;
INT32			nRes, nRet;

	pIF = (LPS7INTERFACE) hIF;
	nRes = pS7_TestInterface(pIF);
	nRet = nRes ? IFM_UNKNOWN : pIF->nMode;
	return nRet;
}
