/********************************************************************
 * Copyright (c) 2021 by SK karoly.saly@matrasoft.hu				*
 *																	*
 * Based on libnodave and SNAP7										*
 * (C) Thomas Hergenhahn (thomas.hergenhahn@web.de) 2002..2005		*
 * (C) Davide Nardella 2013, 2015									*
 *																	*
 * s7com32 - s7isotcp.c												*
 ********************************************************************/
#define WIN32_LEAN_AND_MEAN
#include <skclib32.h>
#include <winsock2.h>
#include <sklibudp.h>
#include <s7com32.h>
#include "private.h"

#pragma comment(lib, "ws2_32.lib")

#pragma pack(push, 1)

#define ISOTCPVERSION				3								// RFC 1006
#define MAX_TSAPLENGTH				16								// Max Length for Src and Dst TSAP
#define MAX_ISOFRAGMENTS			64								// Max fragments

// TPKT Header - ISO on TCP - RFC 1006 (4 bytes)
typedef struct tagTPDUHDR {
	UINT8							nVersion;						// Always 3 for RFC 1006
	UINT8							nReserved;						// 0
	UINT16							nLength;						// Big Endian (entire frame, payload and TPDU included min 7 max 65535)
} TPDUHDR, *LPTPDUHDR;

typedef struct tagCOPT_PARTPDU {
	UINT8							nCode;
	UINT8							nLen;
	UINT8							nValue;
} COPT_PARTPDU, *LPCOPT_PARTPDU;

typedef struct tagCOPT_PARTSAP {
	UINT8							nCode;
	UINT8							nLen;
	UINT16							nValue;
} COPT_PARTSAP, *LPCOPT_PARTSAP;

// PDU Type constants - ISO 8073, not all are mentioned in RFC 1006
// For our purposes we use only those labeled with **
// These constants contains 4 low bit order 0 (credit nibble)
//
//     $10 ED : Expedited Data
//     $20 EA : Expedited Data Ack
//     $40 UD : CLTP UD
//     $50 RJ : Reject
//     $70 AK : Ack data
// **  $80 DR : Disconnect request (note : S7 doesn't use it)
// **  $C0 DC : Disconnect confirm (note : S7 doesn't use it)
// **  $D0 CC : Connection confirm
// **  $E0 CR : Connection request
// **  $F0 DT : Data
//

// COTP Header for CONNECTION REQUEST/CONFIRM - DISCONNECT REQUEST/CONFIRM
typedef struct tagCOTP_CO {
	UINT8							nLength;						// Header length : initialized to 6 (length without params - 1)
																	// descending classes that add values in params field must update it.
	UINT8							nType;							// 0xE0 Connection request
																	// 0xD0 Connection confirm
																	// 0x80 Disconnect request
																	// 0xDC Disconnect confirm
	UINT16							nDstRef;						// Destination reference : Always 0x0000
	UINT16							nSrcRef;						// Source reference : Always 0x0000
	UINT8							nCO_R;							// If the telegram is used for Connection request/Confirm,
																	// the meaning of this field is CLASS+OPTION :
																	//   Class (High 4 bits) + Option (Low 4 bits)
																	//   Class : Always 4 (0100) but is ignored in input (RFC States this)
																	//   Option : Always 0, also this in ignored.
																	// If the telegram is used for Disconnect request,
																	// the meaning of this field is REASON :
																	//    1     Congestion at TSAP
																	//    2     Session entity not attached to TSAP
																	//    3     Address unknown (at TCP connect time)
																	//  128+0   Normal disconnect initiated by the session entity.
																	//  128+1   Remote transport entity congestion at connect request time
																	//  128+3   Connection negotiation failed
																	//  128+5   Protocol Error
																	//  128+8   Connection request refused on this network connection
																	// Parameter data : depending on the protocol implementation.
																	// ISO 8073 define several type of parameters, but RFC 1006 recognizes only
																	// TSAP related parameters and PDU size.  See RFC 0983 for more details.
	COPT_PARTPDU					TPDUSize;
	COPT_PARTSAP					SrcTSAP;
	COPT_PARTSAP					DstTSAP;
} COTP_CO, *LPCOTP_CO;

// COTP Header for DATA EXCHANGE
typedef struct tagCOTP_DT {
	UINT8							nLength;						// Header length : 3 for this header
	UINT8							nType;							// 0xF0 for this header
	UINT8							nEOT;							// EOT (bit 7) + PDU Number (bits 0..6)
																	// EOT = 1 -> End of Trasmission Packet (This packet is complete)
																	// PDU Number : Always 0
} COTP_DT, *LPCOTP_DT;

// Info part of a PDU, only common parts. We use it to check the consistence
// of a telegram regardless of it's nature (control or data).
typedef struct tagISOHEADERINFO {
	TPDUHDR							Hdr;							// TPDU Header
	UINT8							nLength;						// Common part of any COTP: Header length : 3 for this header
	UINT8							nType;							// Common part of any COTP: Pdu type
} ISOHEADERINFO, *LPISOHEADERINFO;

// PDU Type consts (Code + Credit)
#define TPDUT_CR					0xE0							// Connection request
#define TPDUT_CC					0xD0							// Connection confirm
#define TPDUT_DR					0x80							// Disconnect request
#define TPDUT_DC					0xC0							// Disconnect confirm
#define TPDUT_DT					0xF0							// Data transfer

#define TPDUP_TPDU_SIZE    			0xC0							// TPDU Size
#define TPDUP_SRC_TSAP     			0xC1							// TSAP-ID / calling TSAP ( in CR/CC )
#define TPDUP_DST_TSAP     			0xC2							// TSAP-ID / called TSAP

#define TPDUT_EOT					0x80							// End of Trasmission Packet (This packet is complete)

typedef struct tagISOCONTROLTPDU {
	TPDUHDR							Hdr;							// TPKT Header
	COTP_CO							COTP;							// COPT Header for CONNECTION stuffs
} ISOCONTROLTPDU, *LPISOCONTROLTPDU;

typedef struct tagISODATATPDU {
	TPDUHDR							Hdr;							// TPKT Header
	COTP_DT							COTP;							// COPT Header for DATA EXCHANGE
	UINT8							nPayload[MAX_ISOPAYLOADSIZE];
} ISODATATPDU, *LPISODATATPDU;

typedef union tagANYTPDUPTR {
	LPVOID							pVoid;
	LPSTR							pStr;
	LPTPDUHDR						pHdr;
	LPISOHEADERINFO					pHdrInf;
	LPISOCONTROLTPDU				pCtrl;
	LPISODATATPDU					pData;
} ANYTPDUPTR, *LPANYTPDUPTR;

#pragma pack(pop)

#define DATAHEADERSIZE				(sizeof(TPDUHDR) + sizeof(COTP_DT))
#define PG_COMMUNICATION			1
#define OP_COMMUNICATION			2
#define BASIS_COMMUNICATION			3

// **********************************************************************************
static void __stdcall pInitCtrlTPDU(LPS7CONNECTION lpCon, INT32 nFunc) {
ANYTPDUPTR		pSnd;

	pSnd.pStr = lpCon->pSndBuf;
	MemClr(pSnd.pVoid, sizeof(ISOCONTROLTPDU));
	pSnd.pCtrl->Hdr.nVersion = ISOTCPVERSION;
//	pSnd.pCtrl->Hdr.nReserved = 0;
	pSnd.pCtrl->Hdr.nLength = ByteSwap16(sizeof(ISOCONTROLTPDU));
	pSnd.pCtrl->COTP.nLength = sizeof(COTP_CO) - 1;
	pSnd.pCtrl->COTP.nType = nFunc;
//	pSnd.pCtrl->COTP.nDstRef = 0;
	pSnd.pCtrl->COTP.nSrcRef = ByteSwap16(1);
//	pSnd.pCtrl->COTP.nCO_R = 0;
	pSnd.pCtrl->COTP.TPDUSize.nCode = TPDUP_TPDU_SIZE;
	pSnd.pCtrl->COTP.TPDUSize.nLen = 1;
	pSnd.pCtrl->COTP.TPDUSize.nValue = 9;							// 512 = 1 << 9
	pSnd.pCtrl->COTP.SrcTSAP.nCode = TPDUP_SRC_TSAP;
	pSnd.pCtrl->COTP.SrcTSAP.nLen = 2;
	pSnd.pCtrl->COTP.SrcTSAP.nValue = ByteSwap16(0x100);
	pSnd.pCtrl->COTP.DstTSAP.nCode = TPDUP_DST_TSAP;
	pSnd.pCtrl->COTP.DstTSAP.nLen = 2;
	pSnd.pCtrl->COTP.DstTSAP.nValue = ByteSwap16((PG_COMMUNICATION << 8) + (lpCon->nSlot | lpCon->nRack << 5));
};
// **********************************************************************************
static void __stdcall pInitDataTPDU(LPS7CONNECTION lpCon, INT32 nSize) {
ANYTPDUPTR		pSnd;

	pSnd.pStr = lpCon->pSndBuf;
	MemClr(pSnd.pVoid, DATAHEADERSIZE);
	pSnd.pData->Hdr.nVersion = ISOTCPVERSION;
//	pSnd.pData->Hdr.nReserved = 0;
	pSnd.pData->Hdr.nLength = ByteSwap16(DATAHEADERSIZE + nSize);
	pSnd.pData->COTP.nLength = sizeof(COTP_DT) - 1;
	pSnd.pData->COTP.nType = TPDUT_DT;
	pSnd.pData->COTP.nEOT = 0x80;
};
// **********************************************************************************
static INT32 __stdcall pRequest(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
ANYTPDUPTR			pSnd, pRcv;
INT32				nRes, nErr, nRetry;
UINT32				nStrt, nAct, nRcvLen, nRead, nReady, nSize, nType, nTotalLen;
BOOL				bWait, bFollow;
ISODATATPDU			tpdu;

	pSnd.pStr = lpCon->pSndBuf;
	pRcv.pStr = lpCon->pRcvBuf;
	MemClr(&tpdu, sizeof(ISOHEADERINFO));
	MemClr(pRcv.pStr, sizeof(ISOHEADERINFO));
	nRetry = 0;
	nSize = ByteSwap16(pSnd.pHdrInf->Hdr.nLength);
	do {
		nRes = send(lpCon->NetInf.nSkt, pSnd.pStr, nSize, 0);
		if (nRes != nSize) {
			nErr = WSAGetLastError();
			nRes = S7ERR_SOCKETERROR;
		}
		else {
			nRes = nErr = 0;
		}
		nRetry++;
	} while (nRes && (nRetry < 3));
	if (!nRes && (pSnd.pHdrInf->nType != TPDUT_DR)) {
		nTotalLen = 0;
		switch (pSnd.pHdrInf->nType) {
		case TPDUT_CR:
			nType = TPDUT_CC;
			break;
		case TPDUT_DR:
			nType = TPDUT_DC;
			break;
		default:
			nType = TPDUT_DT;
		}
		do {
			bFollow = FALSE;
			nStrt = GetTickCount();
			bWait = TRUE;
			do {
				nRcvLen = 0;
				nRes = ioctlsocket(lpCon->NetInf.nSkt, FIONREAD, &nRcvLen);
				if (nRes) {
					nErr = WSAGetLastError();
					nRes = S7ERR_SOCKETERROR;
				}
				if (nRes || nRcvLen) {
					bWait = FALSE;
				}
				else {
					Sleep(0);
					nAct = GetTickCount();
					if ((nAct - nStrt) >= lpCon->nTimeOut) {
						bWait = FALSE;
						nRes = S7ERR_TIMEOUT;
					}
				}
			} while (bWait);
			if (!nRes && ((nRcvLen < DATAHEADERSIZE) || (nRcvLen > sizeof(ISODATATPDU)))) {
					nRes = S7ERR_INVALIDRESPONSE;
			}
			if (!nRes) {
				nRead = pSnd.pHdrInf->nType == TPDUT_DT ? DATAHEADERSIZE : nRcvLen;
				nReady = recv(lpCon->NetInf.nSkt, (LPSTR) &tpdu, nRead, 0);
				nSize = ByteSwap16(tpdu.Hdr.nLength);
				if (nReady != nRead) {
					nErr = WSAGetLastError();
					nRes = S7ERR_SOCKETERROR;
				}
				else if ((tpdu.Hdr.nVersion != ISOTCPVERSION) || tpdu.Hdr.nReserved) {
					nRes = S7ERR_INVALIDRESPONSE;
				}
				else {
					if (pSnd.pHdrInf->nType == TPDUT_DT) {
						nRead = nSize - DATAHEADERSIZE;
						nRcvLen -= DATAHEADERSIZE;
						if (nRead) {
							if (nRead <= nRcvLen) {
								nReady = recv(lpCon->NetInf.nSkt, (LPSTR) &tpdu.nPayload, nRead, 0);
								if (nReady != nRead) {
									nErr = WSAGetLastError();
									nRes = S7ERR_SOCKETERROR;
								}
							}
							else {
								nRes = S7ERR_INVALIDRESPONSE;
							}
						}
						bFollow = (tpdu.COTP.nEOT & 0x80) == 0;
					}
					if (!nRes && (tpdu.COTP.nType != nType)) nRes = S7ERR_REJECTEDRESPONSE;
				}
			}
			else {
				while (nRcvLen) {
					nAct = nRcvLen > sizeof(ISODATATPDU) ? sizeof(ISODATATPDU) : nRcvLen;
					nRcvLen -= nAct;
					recv(lpCon->NetInf.nSkt, (LPSTR) &tpdu, nAct, 0);
				}
			}
			if (!nRes) {
				if (nType == TPDUT_DT) {
					if (!nTotalLen) nTotalLen = DATAHEADERSIZE;
					MemCpy(pRcv.pStr, &tpdu, DATAHEADERSIZE);
					MemCpy(pRcv.pStr + nTotalLen, &tpdu.nPayload, nSize - DATAHEADERSIZE);
					nTotalLen += nSize - DATAHEADERSIZE;
				}
				else {
					MemCpy(pRcv.pStr, &tpdu, nSize);
					nTotalLen = nSize;
				}
			}
		} while (bFollow & !nRes);
	}
	*lpSysErr = nErr;
	return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7IsoTcp_Connect(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
ANYTPDUPTR		pRcv;
INT32			nRes;

	lpCon->pSndBuf = GetMem(GMM_DEFAULT, 2 * (MAX_S7PDUSIZE + DATAHEADERSIZE));
	if (!lpCon->pSndBuf) return S7ERR_NOMEMORY;
	lpCon->pRcvBuf = lpCon->pSndBuf + MAX_S7PDUSIZE + DATAHEADERSIZE;
	lpCon->SndPdu.pData = lpCon->pSndBuf + DATAHEADERSIZE;
	lpCon->RcvPdu.pData = lpCon->pRcvBuf + DATAHEADERSIZE;
	nRes = OpenSocket(NULL, 0, ISO_PORT, SOCK_STREAM, IPPROTO_IP, FALSE, FALSE, 50, 50, &lpCon->NetInf, NULL, lpSysErr);
	if (nRes) return S7ERR_SOCKETERROR;
	nRes = TcpConnect(lpCon->NetInf.nSkt, lpCon->nAddr, ISO_PORT, 2000, lpSysErr);
	if (nRes) {
		return nRes == -1 ? S7ERR_TIMEOUT : S7ERR_SOCKETERROR;
	}
	pInitCtrlTPDU(lpCon, TPDUT_CR);
	nRes = pRequest(lpCon, lpSysErr);
	if (nRes) return nRes;
	pRcv.pStr = lpCon->pRcvBuf;
	lpCon->nTPDUSize = 1 << pRcv.pCtrl->COTP.TPDUSize.nValue;
	pS7PDU_SndInit(lpCon, PDUT_REQUEST, PDUFN_NEGOTIATE, 0, FALSE);
	pInitDataTPDU(lpCon, lpCon->SndPdu.nTotLen);
	nRes = pRequest(lpCon, lpSysErr);
	if (nRes) return nRes;
    nRes = pS7PDU_RcvInit(lpCon, lpSysErr);
	if (nRes) return nRes;
	nRes = pS7PDU_SetPDUSize(lpCon);
	return nRes;
};
// **********************************************************************************
INT32 __stdcall pS7IsoTcp_Disconnect(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
INT32			nRes;

	pInitCtrlTPDU(lpCon, TPDUT_DR);
	nRes = pRequest(lpCon, lpSysErr);
	CloseSocket(lpCon->NetInf.nSkt, lpCon->NetInf.bWSA);
	return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7IsoTcp_Exchange(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
INT32	nRes;

	pInitDataTPDU(lpCon, lpCon->SndPdu.nTotLen);
	nRes = pRequest(lpCon, lpSysErr);
    return nRes;
}
