/***
*gs_report.c - Report a /GS security check failure
*
*       Copyright (c) Microsoft Corporation.  All rights reserved.
*
*Purpose:
*       Defines function used to report a security check failure.
*       This file corresponds to gs_report.c in the Windows sources.
*
*       Entrypoints:
*            __report_gsfailure
*
*******************************************************************************/

#include "crtdefs.h"
#include "crtplus.h"

#if defined (_CRTBLD)
#include "dbgint.h"					// needed for _CRT_DEBUGGER_HOOK
#endif  // defined (_CRTBLD)

typedef LONG NTSTATUS;
// ***SK #define STATUS_STACK_BUFFER_OVERRUN      ((NTSTATUS)0xC0000409L)
// from gs_support
// SK typedef
// EXCEPTION_DISPOSITION
// (*PEXCEPTION_ROUTINE) (
//     IN struct _EXCEPTION_RECORD *ExceptionRecord,
//     IN PVOID EstablisherFrame,
//     IN OUT struct _CONTEXT *ContextRecord,
//     IN OUT PVOID DispatcherContext
//     );

typedef struct _EXCEPTION_REGISTRATION_RECORD {
    struct _EXCEPTION_REGISTRATION_RECORD *Next;
    PEXCEPTION_ROUTINE Handler;
} EXCEPTION_REGISTRATION_RECORD;

typedef EXCEPTION_REGISTRATION_RECORD *PEXCEPTION_REGISTRATION_RECORD;

#define EXCEPTION_CHAIN_END ((struct _EXCEPTION_REGISTRATION_RECORD * POINTER_32)-1)

EXCEPTION_DISPOSITION __cdecl
_except_handler4(
    IN struct _EXCEPTION_RECORD *ExceptionRecord,
    IN PVOID EstablisherFrame,
    IN OUT struct _CONTEXT *ContextRecord,
    IN OUT PVOID DispatcherContext
    );
// from gs_support-end

extern UINT_PTR __security_cookie;
extern UINT_PTR __security_cookie_complement;

/*
 * Use global memory for the exception and context records, and any other local
 * state needed by __report_gsfailure.  We're not worried about multithread
 * issues here - we're going to die anyway, and it saves us from consuming a
 * good chunk of stack, as well as potentially overwriting useful data in the
 * stack memory dump.
 */

static EXCEPTION_RECORD         GS_ExceptionRecord;
static CONTEXT                  GS_ContextRecord;
static const EXCEPTION_POINTERS GS_ExceptionPointers = {
    &GS_ExceptionRecord,
    &GS_ContextRecord
};

#if defined (_CRTBLD)
static BOOL DebuggerWasPresent;
#endif  // defined (_CRTBLD)

/***
*__report_gsfailure() - Report security error.
*
*Purpose:
*       A /GS security error has been detected.  We save the registers in a
*       CONTEXT struct & call UnhandledExceptionFilter to display a message
*       to the user (invoke Watson handling) and terminate the program.
*
*       NOTE: __report_gsfailure should not be called directly.  Instead, it
*       should be entered due to a failure detected by __security_check_cookie.
*       That's because __security_check_cookie may perform more checks than
*       just a mismatch against the global security cookie, and because
*       the context captured by __report_gsfailure is unwound assuming that
*       __security_check_cookie was used to detect the failure.
*
*Entry:
*       ULONGLONG StackCookie - the local cookie, passed in as actual argument
*       on AMD64 and IA64 only.  On x86, the local cookie is available in ECX,
*       but since __report_gsfailure isn't __fastcall, it isn't a true
*       argument, and we must flush ECX to the context record quickly.
*
*Exit:
*       Does not return.
*
*Exceptions:
*
*******************************************************************************/

#pragma optimize("", off)   /* force an EBP frame on x86, no stack packing */

#if defined (_X86_)
__declspec(noreturn) void __cdecl __report_gsfailure(void)
#else  /* defined (_X86_) */
__declspec(noreturn) void __cdecl __report_gsfailure(ULONGLONG StackCookie)
#endif  /* defined (_X86_) */
{
    volatile UINT_PTR cookie[2];

    /*
     * Set up a fake exception, and report it via UnhandledExceptionFilter.
     * We can't raise a true exception because the stack (and therefore
     * exception handling) can't be trusted after a buffer overrun.  The
     * exception should appear as if it originated after the call to
     * __security_check_cookie, so it is attributed to the function where the
     * buffer overrun was detected.
     */

    /*
     * On x86, we reserve some extra stack which won't be used.  That is to
     * preserve as much of the call frame as possible when the function with
     * the buffer overrun entered __security_check_cookie with a JMP instead of
     * a CALL, after the calling frame has been released in the epilogue of
     * that function.
     */

    volatile ULONG dw[(sizeof(CONTEXT) + sizeof(EXCEPTION_RECORD))/sizeof(ULONG)];

    /*
     * Save the state in the context record immediately.  Hopefully, since
     * opts are disabled, this will happen without modifying ECX, which has
     * the local cookie which failed the check.
     */

    __asm {
        mov dword ptr [GS_ContextRecord.Eax], eax
        mov dword ptr [GS_ContextRecord.Ecx], ecx
        mov dword ptr [GS_ContextRecord.Edx], edx
        mov dword ptr [GS_ContextRecord.Ebx], ebx
        mov dword ptr [GS_ContextRecord.Esi], esi
        mov dword ptr [GS_ContextRecord.Edi], edi
        mov word ptr [GS_ContextRecord.SegSs], ss
        mov word ptr [GS_ContextRecord.SegCs], cs
        mov word ptr [GS_ContextRecord.SegDs], ds
        mov word ptr [GS_ContextRecord.SegEs], es
        mov word ptr [GS_ContextRecord.SegFs], fs
        mov word ptr [GS_ContextRecord.SegGs], gs
        pushfd
        pop [GS_ContextRecord.EFlags]

        /*
         * Set the context EBP/EIP/ESP to the values which would be found
         * in the caller to __security_check_cookie.
         */

        mov eax, [ebp]
        mov dword ptr [GS_ContextRecord.Ebp], eax
        mov eax, [ebp+4]
        mov dword ptr [GS_ContextRecord.Eip], eax
        lea eax, [ebp+8]
        mov dword ptr [GS_ContextRecord.Esp], eax

        /*
         * Make sure the dummy stack space looks referenced.
         */

        mov eax, dword ptr dw
    }

    GS_ContextRecord.ContextFlags = CONTEXT_CONTROL;

    GS_ExceptionRecord.ExceptionAddress  = (PVOID)(ULONG_PTR)GS_ContextRecord.Eip;

    GS_ExceptionRecord.ExceptionCode  = STATUS_STACK_BUFFER_OVERRUN;
    GS_ExceptionRecord.ExceptionFlags = EXCEPTION_NONCONTINUABLE;

    /*
     * Save the global cookie and cookie complement locally - using an array
     * to defeat any potential stack-packing.
     */

    cookie[0] = __security_cookie;
    cookie[1] = __security_cookie_complement;

    DebuggerWasPresent = IsDebuggerPresent();
    _crt_debugger_hook(_CRT_DEBUGGER_GSFAILURE);

    /* Make sure any filter already in place is deleted. */
    SetUnhandledExceptionFilter(NULL);

    UnhandledExceptionFilter((EXCEPTION_POINTERS *)&GS_ExceptionPointers);

#if defined (_CRTBLD)
    /*
     * If we make it back from Watson, then the user may have asked to debug
     * the app.  If we weren't under a debugger before invoking Watson,
     * re-signal the VS CRT debugger hook, so a newly attached debugger gets
     * a chance to break into the process.
     */
    if (!DebuggerWasPresent)
    {
        _crt_debugger_hook(_CRT_DEBUGGER_GSFAILURE);
    }
#endif  // defined (_CRTBLD)

    TerminateProcess(GetCurrentProcess(), STATUS_STACK_BUFFER_OVERRUN);
}
