/***
*onexit.c - save function for execution on exit
*
*       Copyright (c) Microsoft Corporation. All rights reserved.
*
*Purpose:
*       defines _onexit(), atexit() - save function for execution at exit
*
*       In order to save space, the table is allocated via malloc/realloc,
*       and only consumes as much space as needed.  __onexittable is
*       set to point to the table if onexit() is ever called.
*
*******************************************************************************/

#include "crtdefs.h"
#define ENABLE_STDLIB_H
#include "crtplus.h"
#include "libcmt.h"
#include "sect_attribs.h"
#include "cruntime.h"
#include "internal.h"
#include "malloc.h"
#include "dbgint.h"


int __cdecl __onexitinit(void);
void __cdecl __onexitdone(void);

_CRTALLOC(".CRT$XIB") static _PIFV pinit = __onexitinit;
_CRTALLOC(".CRT$XTB") static _PVFV pdone = __onexitdone;

/*
 * Define pointers to beginning and end of the table of function pointers
 * manipulated by _onexit()/atexit().
 * NOTE - the pointers are stored encoded.
 */
LP_PVFV		__onexitbegin = NULL;
LP_PVFV		__onexitend = NULL;

/*
 * Define increments (in entries) for growing the _onexit/atexit table
 */
#define MININCR     4
#define MAXINCR     512

static _onexit_t __cdecl _onexit_nolock(_onexit_t);
static _onexit_t __cdecl _dllonexit_nolock(_onexit_t, _PVFV **, _PVFV **);

/***
*_onexit(func), atexit(func) - add function to be executed upon exit
*
*Purpose:
*       The _onexit/atexit functions are passed a pointer to a function
*       to be called when the program terminate normally.  Successive
*       calls create a register of functions that are executed last in,
*       first out.
*
*Entry:
*       void (*func)() - pointer to function to be executed upon exit
*
*Exit:
*       onexit:
*           Success - return pointer to user's function.
*           Error - return NULL pointer.
*       atexit:
*           Success - return 0.
*           Error - return non-zero value.
*
*Notes:
*       This routine depends on the behavior of _initterm() in CRT0DAT.C.
*       Specifically, _initterm() must not skip the address pointed to by
*       its first parameter, and must also stop before the address pointed
*       to by its second parameter.  This is because _onexitbegin will point
*       to a valid address, and _onexitend will point at an invalid address.
*
*Exceptions:
*
*******************************************************************************/

_onexit_t __cdecl _onexit (
        _onexit_t func
        )
{
        _onexit_t retval;

        _lock(_EXIT_LOCK);

        __try {
            retval = _onexit_nolock(func);
        }
        __finally {
            _unlock(_EXIT_LOCK);
        }

        return retval;
}


static _onexit_t __cdecl _onexit_nolock (
        _onexit_t func
        )
{
        _PVFV * p;
        size_t  oldsize;
        _PVFV * onexitbegin = __onexitbegin;
        _PVFV * onexitend = __onexitend;

        /* overflow check */
        if (onexitend < onexitbegin ||
            ((char *)onexitend - (char *)onexitbegin) + sizeof(_PVFV) < sizeof(_PVFV))
        {
            return NULL;
        }

        /*
         * First, make sure the table has room for a new entry
         */
        if ( (oldsize = _msize(onexitbegin))
                < ((size_t)((char *)onexitend -
            (char *)onexitbegin) + sizeof(_PVFV)) )
        {
            /*
             * not enough room, try to grow the table. first, try to double it.
             */
            size_t newsize = oldsize + __min(oldsize, (MAXINCR * sizeof(_PVFV)));
            if ( newsize < oldsize ||
                 (p = (_PVFV *)realloc(onexitbegin, newsize)) == NULL )
            {
                /*
                 * failed, try to grow by MININCR
                 */
                newsize = oldsize + MININCR * sizeof(_PVFV);
                if ( newsize < oldsize ||
                     (p = (_PVFV *)realloc(onexitbegin, newsize)) == NULL )
                    /*
                     * failed again. don't do anything rash, just fail
                     */
                    return NULL;
            }

            /*
             * update __onexitend and __onexitbegin
             */
#pragma warning(suppress: 22008) /* prefast is confused */
            onexitend = p + (onexitend - onexitbegin);
            onexitbegin = p;
            __onexitbegin = onexitbegin;
        }

        /*
         * Put the new entry into the table and update the end-of-table
         * pointer.
         */
         *(onexitend++) = func;
        __onexitend = onexitend;

        return func;
}

int __cdecl atexit (
        _PVFV func
        )
{
        return (_onexit((_onexit_t)func) == NULL) ? -1 : 0;
}


/***
* void __onexitinit(void) - initialization routine for the function table
*       used by _onexit() and atexit().
*
*Purpose:
*       Allocate the table with room for 32 entries (minimum required by
*       ANSI). Also, initialize the pointers to the beginning and end of
*       the table.
*
*Entry:
*       None.
*
*Exit:
*       Returns _RTERR_ONEXIT if the table cannot be allocated.
*
*Notes:
*       This routine depends on the behavior of doexit() in CRT0DAT.C.
*       Specifically, doexit() must not skip the address pointed to by
*       __onexitbegin, and it must also stop before the address pointed
*       to by __onexitend.  This is because _onexitbegin will point
*       to a valid address, and _onexitend will point at an invalid address.
*
*       Since the table of onexit routines is built in forward order, it
*       must be traversed by doexit() in CRT0DAT.C in reverse order.  This
*       is because these routines must be called in last-in, first-out order.
*
*       If __onexitbegin == __onexitend, then the onexit table is empty!
*
*Exceptions:
*
*******************************************************************************/

int __cdecl __onexitinit (
        void
        )
{
        _PVFV * onexitbegin;

        onexitbegin = (_PVFV *)calloc(32, sizeof(_PVFV));
        __onexitend = __onexitbegin = onexitbegin;

        if ( onexitbegin == NULL )
            /*
             * cannot allocate minimal required size. return
             * fatal runtime error.
             */
            return _RTERR_ONEXIT;

        *onexitbegin = (_PVFV) NULL;

        return 0;
}


//	do _onexit/atexit() terminators (if there are any)
//	These terminators MUST be executed in reverse order (LIFO)!
//	NOTE: This code assumes that __onexitbegin points to the first valid onexit() entry and that
//	__onexitend points past the last valid entry.
//	If __onexitbegin == __onexitend, the table is empty and there are no routines to call.
void __cdecl __onexitdone(void) {
LP_PVFV	onexitbegin, onexitend, onexitbegin_saved, onexitend_saved, onexitbegin_new, onexitend_new;
_PVFV	function_to_call;

	onexitbegin = __onexitbegin;
	if (onexitbegin) {
		onexitend = __onexitend;
		function_to_call = NULL;
		onexitbegin_saved = onexitbegin;
		onexitend_saved = onexitend;
		while (1) {
			onexitbegin_new = onexitend_new = NULL;
//	find the last valid function pointer to call.
			while ((--onexitend >= onexitbegin) && (*onexitend == NULL)) {	//keep going backwards.
			}
			if (onexitend < onexitbegin) {
				break;		//	there are no more valid entries in the list, we are done.
			}
			function_to_call = *onexitend;		//	cache the function to call.
			*onexitend = NULL;					//	mark the function pointer as visited.
			(*function_to_call)();				//call the function, which can eventually change __onexitbegin and __onexitend
			onexitbegin_new = __onexitbegin;
			onexitend_new = __onexitend;
			if ((onexitbegin_saved != onexitbegin_new) || (onexitend_saved != onexitend_new)) {
				onexitbegin = onexitbegin_saved = onexitbegin_new;
				onexitend = onexitend_saved = onexitend_new;	// reset only if either start or end has changed
			}
		}
	}
	FreeMemEx(&(LPVOID)__onexitbegin);
}
