/***
*setenv.c -set an environment variable in the environment
*
*       Copyright (c) Microsoft Corporation. All rights reserved.
*
*Purpose:
*       defines __crtsetenv() - adds a new variable to environment.
*       Internal use only.
*
*******************************************************************************/


#include "crtdefs.h"
#define ENABLE_STRING_H
#define ENABLE_STDLIB_H
#include "crtplus.h"
#include "libcmt.h"
#include "cruntime.h"
#include "internal.h"
#include "tchar.h"
#include "dbgint.h"
#include "awint.h"
#include "limits.h"

static int __cdecl findenv(const char *name, int len);

/***
*int __crtsetenv(option) - add/replace/remove variable in environment
*
*Purpose:
*       option should be of the form "option=value".  If a string with the
*       given option part already exists, it is replaced with the given
*       string; otherwise the given string is added to the environment.
*       If the string is of the form "option=", then the string is
*       removed from the environment, if it exists.  If the string has
*       no equals sign, error is returned.
*
*Entry:
*       TCHAR **poption - pointer to option string to set in the environment list.
*           should be of the form "option=value".
*           This function takes ownership of this pointer in the success case.
*       int primary - Only the primary call to _crt[w]setenv needs to
*           create new copies or set the OS environment.
*           1 indicates that this is the primary call.
*
*Exit:
*       returns 0 if OK, -1 if fails.
*       If *poption is non-null on exit, we did not free it, and the caller should
*       If *poption is null on exit, we did free it, and the caller should not.
*
*Exceptions:
*
*Warnings:
*       This code will not work if variables are removed from the environment
*       by deleting them from environ[].  Use _putenv("option=") to remove a
*       variable.
*
*       The option argument will be taken ownership of by this code and may be freed!
*
*******************************************************************************/

int __cdecl __crtsetenv (
        char **poption,
        const int primary
        )
{
        int ix;
        int retval = 0;
        int remove; /* 1 if variable is to be removed */
        char *option=NULL;           /* The option string passed in */
        char **env;
        char *name, *value;
        const char *equal;

        /* Validate poption and dereference it */
        _VALIDATE_RETURN(poption != NULL, EINVAL, -1);

        option=*poption;

        /*
         * check that the option string is valid, find the equal sign
         * and verify '=' is not the first character in string.
         */
        if ( (option == NULL) || ((equal = _tcschr(option, _T('='))) == NULL)
            || option == equal)
        {
            errno = EINVAL;
            return -1;
        }

        /* internal consistency check: the environment string should never use buffers bigger than _MAX_ENV
         * see also SDK function SetEnvironmentVariable
         */
        _ASSERTE(equal - option < _MAX_ENV);
        _ASSERTE(_tcsnlen(equal + 1, _MAX_ENV) < _MAX_ENV);

        /* if the character following '=' is null, we are removing the
         * the environment variable. Otherwise, we are adding or updating
         * an environment variable.
         */
        remove = (*(equal + 1) == _T('\0'));

        /* see if requested environment array exists */
        if (_environ == NULL) {

            /*
             * The requested type of environment does not exist.
             * See if other type exists, if so convert it to requested type.
             * The functions that convert the enviroment (__mbtow_environ and
             * __wtomb_environ) will call this function (__crt[w]setenv) once
             * for each of the pre-existing environment variables. To avoid
             * an infinite loop, test the primary flag.
             */

                /* nothing to remove, return */
                if ( remove )
                    return 0;
                else {
                    /* create ones that do not exist */

                    if (_environ == NULL)
                    {
                        if ( (_environ = malloc(sizeof(char *))) == NULL)
                            return -1;
                        *_environ = NULL;
                    }

                }
        }

        /*
         * At this point, the two types of environments are in sync (as much
         * as they can be anyway). The only way they can get out of sync
         * (besides users directly modifiying the environment) is if there
         * are conversion problems: If the user sets two Unicode EVs,
         * "foo1" and "foo2" and converting then to multibyte yields "foo?"
         * and "foo?", then the environment blocks will differ.
         */

        /* init env pointers */
        env = _environ;
                if(!env)
                {
                        _ASSERTE(("CRT Logic error during setenv",0));
                        return -1;
                }

        /* See if the string is already in the environment */
        ix = findenv(option, (int)(equal - option));

        if ((ix >= 0) && (*env != NULL)) {
            /*
             * String is already in the environment. Free up the original
             * string. Then, install the new string or shrink the environment,
             * whichever is warranted.
             */
            free(env[ix]);

            if (remove) {

                /* removing -- move all the later strings up */
                for ( ; env[ix] != NULL; ++ix) {
                    env[ix] = env[ix+1];
                }

                /* shrink the environment memory block
                   (ix now has number of strings, including NULL) --
                   this realloc probably can't fail, since we're
                   shrinking a mem block, but we're careful anyway. */
                if (ix < (SIZE_MAX / sizeof(char *)) &&
                    (env = (char **) _recalloc(_environ, ix, sizeof(char *))) != NULL)
                    _environ = env;
            }
            else {
                /* replace the option */
                env[ix] = (char *) option;

                /* we now own the pointer, so NULL out the incoming pointer */
                *poption=NULL;
            }
        }
        else {
            /*
             * String is NOT in the environment
             */
            if ( !remove )  {
                /*
                 * Append the string to the environ table. Note that
                 * table must be grown to do this.
                 */
                if (ix < 0)
                    ix = -ix;    /* ix = length of environ table */

                if ((ix + 2) < ix ||
                    (ix + 2) >= (SIZE_MAX / sizeof(char *)) ||
                    (env = (char **)_recalloc(_environ, sizeof(char *), (ix + 2))) == NULL)
                    return -1;

                env[ix] = (char *)option;
                env[ix + 1] = NULL;

                /* we now own the pointer, so NULL out the incoming pointer */
                *poption=NULL;

                _environ = env;
            }
            else {
                /*
                 * We are asked to remove an environment var that isn't there.
                 * Free the option string and return success.
                 */
                free(option);

                /* we now freed the pointer, so NULL out the incoming pointer */
                *poption=NULL;

                return 0;
            }
        }

        /*
         * Update the OS environment. Don't give an error if this fails
         * since the failure will not affect the user unless he/she is making
         * direct API calls. Only need to do this for one type, OS converts
         * to other type automatically.
         */
        if ( primary &&
            (name = (char *)calloc((_tcslen(option) + 2), sizeof(char))) != NULL )
        {
            _ERRCHECK(_tcscpy_s(name, _tcslen(option) + 2, option));
            value = name + (equal - option);
            *value++ = _T('\0');
            if (SetEnvironmentVariable(name, remove ? NULL : value) == 0)
                retval = -1;
            if (retval == -1)
            {
                errno = EILSEQ;
            }

            free(name);
        }

        if (remove) {
            /* free option string since it won't be used anymore */
            free(option);
            *poption = NULL;
        }

        return retval;
}


/***
*int findenv(name, len) - [STATIC]
*
*Purpose:
*       Scan for the given string within the environment
*
*Entry:
*
*Exit:
*       Returns the offset in "environ[]" of the given variable
*       Returns the negative of the length of environ[] if not found.
*       Returns 0 if the environment is empty.
*
*       [NOTE: That a 0 return can mean that the environment is empty
*       or that the string was found as the first entry in the array.]
*
*Exceptions:
*
*******************************************************************************/

#ifdef WPRFLAG
static int __cdecl wfindenv (
#else  /* WPRFLAG */
static int __cdecl findenv (
#endif  /* WPRFLAG */
        const char *name,
        int len
        )
{
        char **env;

        for ( env = _environ; *env != NULL ; env++ ) {
            /*
             * See if first len characters match, up to case
             */
            if ( _tcsnicoll(name, *env, len) == 0 )
                /*
                 * the next character of the environment string must
                 * be an '=' or a '\0'
                 */
                if ( (*env)[len] == _T('=') || (*env)[len] == _T('\0') )
                    return(int)(env - _environ);
//
// We cannot break here since findenv must report the total number of strings.
//              else
//                  break;
        }

        return(-(int)(env - _environ));
}
