/***
*wstdenvp.c - standard _setenvp routine (wchar_t version)
*
*       Copyright (c) Microsoft Corporation. All rights reserved.
*
*Purpose:
*       This module is called by the C start-up routine to set up "_wenviron".
*       Its sets up an array of pointers to strings in the environment.
*       The global symbol "_wenviron" is set to point to this array.
*
*******************************************************************************/


#define WPRFLAG 1

#ifndef _UNICODE
#define _UNICODE 1
#endif  /* _UNICODE */

#ifndef UNICODE
#define UNICODE 1
#endif  /* UNICODE */

/***
*stdenvp.c - standard _setenvp routine
*
*       Copyright (c) Microsoft Corporation. All rights reserved.
*
*Purpose:
*       This module is called by the C start-up routine to set up "_wenviron".
*       Its sets up an array of pointers to strings in the environment.
*       The global symbol "_wenviron" is set to point to this array.
*
*******************************************************************************/

#include "crtdefs.h"
#define ENABLE_STRING_H
#define ENABLE_STDLIB_H
#include "crtplus.h"
#include "libcmt.h"
#include "cruntime.h"
#include "internal.h"
#include "oscalls.h"
#include "tchar.h"
#include "dbgint.h"
#include "sect_attribs.h"

/*
 * Flag checked by getenv() and _putenv() to determine if the environment has
 * been initialized.
 */
wchar_t		**_wenviron = NULL;

static int __cdecl __EnvInitW(void);
#ifdef _DEBUG
static void __cdecl __EnvTermW(void);
#endif	// _DEBUG

_CRTALLOC(".CRT$XIC") static _PIFV _pInit = __EnvInitW;
#ifdef _DEBUG
_CRTALLOC(".CRT$XTC") static _PVFV _pTerm = __EnvTermW;
#endif	// _DEBUG

/***
*_setenvp - set up "envp" for C programs
*
*Purpose:
*       Reads the environment and build the envp array for C programs.
*
*Entry:
*       The environment strings occur at _aenvptr.
*       The list of environment strings is terminated by an extra null
*       byte.  Thus two null bytes in a row indicate the end of the
*       last environment string and the end of the environment, resp.
*
*Exit:
*       "environ" points to a null-terminated list of pointers to ASCIZ
*       strings, each of which is of the form "VAR=VALUE".  The strings
*       are copied from the environment area. This array of pointers will
*       be malloc'ed.  The block pointed to by _aenvptr is deallocated.
*
*Uses:
*       Allocates space on the heap for the environment pointers.
*
*Exceptions:
*       If space cannot be allocated, program is terminated.
*
*******************************************************************************/

static int __cdecl __EnvInitW (
        void
        )
{
LPWSTR p, envptr;
        wchar_t **env;              /* _environ ptr traversal pointer */
        int numstrings;             /* number of environment strings */
        int cchars;

        numstrings = 0;

		p = envptr = GetEnvironmentStrings();				// get wide environ info

        /*
         * We called __crtGetEnvironmentStrings[AW] just before this,
         * so if _[aw]envptr is NULL, we failed to get the environment.
         * Return an error.
         */
        if (p == NULL)
            return _RTERR_SPACEENV;

        /*
         * NOTE: starting with single null indicates no environ.
         * Count the number of strings. Skip drive letter settings
         * ("=C:=C:\foo" type) by skipping all environment variables
         * that begin with '=' character.
         */

        while (*p != _T('\0')) {
            /* don't count "=..." type */
            if (*p != _T('='))
                ++numstrings;
            p += _tcslen(p) + 1;
        }

        /* need pointer for each string, plus one null ptr at end */
        if ( (_wenviron = env = (wchar_t **)
            calloc((numstrings+1), sizeof(wchar_t *))) == NULL )
            return -1;

        /* copy strings to malloc'd memory and save pointers in _wenviron */
        for ( p = envptr ; *p != L'\0' ; p += cchars )
        {
            cchars = (int)_tcslen(p) + 1;
            /* don't copy "=..." type */
            if (*p != _T('=')) {
                if ( (*env = (wchar_t *)calloc(cchars, sizeof(wchar_t)))
                     == NULL )
                {
                    free(_wenviron);
                    _wenviron = NULL;
                    return _RTERR_SPACEENV;
                }
                _ERRCHECK(_tcscpy_s(*env, cchars, p));
                env++;
            }
        }

        /* and a final NULL pointer */
        *env = NULL;


        return 0;
}
//*********************************************************************
#ifdef _DEBUG
static void __cdecl __EnvTermW() {
LPWSTR	*pptr;

	if (_wenviron) {
		for (pptr = _wenviron; *pptr != NULL; ++pptr) {
			free(*pptr);
		}
		FreeMemEx(&(LPVOID) _wenviron);
	}
}
#endif	// _DEBUG
