/********************************************************************
 * Copyright (c) 2021 by SK karoly.saly@matrasoft.hu				*
 *																	*
 * Based on libnodave												*
 * (C) Thomas Hergenhahn (thomas.hergenhahn@web.de) 2002..2005		*
 *																	*
 * s7com32 - s7online.c												*
 ********************************************************************/
#define WIN32_LEAN_AND_MEAN
#include <skclib32.h>
#include <s7com32.h>
#include "private.h"

#define FLC_SUBSYS					0x22
#define TCL_SUBSYS					0x40							// Transport Control Layer

#define COM_REQUEST					0x00
#define COM_CONFIRM					0x01
#define COM_INDICATION              0x02

#define FDL_READ_VALUE				0x0b							// Read busparameter
#define FDL_LIFE_LIST_CREATE_LOCAL	0x1b							// Requests quick-list of intact stations (LAS and GAP will be actual)

#define PRIORITY_LOW				0x00
#define PRIORITY_HIGH				0x01

#define OPEN_REQ					0x00
#define SEND_CONN_REQ				0x01
#define SEND_EOM_DATA				0x06
#define RECEIVE_DATA				0x07
#define CLOSE_REQ					0x0C
#define REQ_WAIT					0xFF

#define OK_RESP						1
#define OK_EOM_RESP					3
/*
#define OK_DECIDE_REQ_RESP			5
#define OK_CLOSED_RESP				7
#define OK_REJECT_CONN_RESP			11
#define INVALID_REQ					2
#define NO_RESOURCES				4
#define UNKNOWN_REFERENCE			6
#define BUFFER_TOO_SHORT			8
#define BUFFER_TOO_LONG				10
#define ILLEGAL_REQ					12
#define REM_ABORT					14
#define LOC_TIMEOUT					16
#define UNKNOWN_CONN_CLASS			18
#define DUP_REQ						20
#define CONN_REJECT					22
#define NEGOT_FAILED				24
#define ILLEGAL_ADDRESS				26
#define NETWORK_ERROR				28
#define PROTOCOL_ERR				30
#define ILLEGAL_RB_LENGTH			32
*/

#pragma pack(push, 1)

typedef struct tagFLCAPPBLK {
	UINT8							nOpCode;						// class of communication
	UINT8							nSubSystem;						// number of source-task (only necessary for MTK-user !!!!!)
	UINT16							nID;							// identification of FDL-USER
	INT16							nServiceCode;					// identification of service
	UINT8							nReserved06[40];
} FLCAPPBLK, *LPFLCAPPBLK;

typedef struct tagTCLAPPBLK {
	UINT16							nReference;						// OPEN_REQ return value
	UINT8							nReserved02[6];
	UINT8							nField08;
	UINT8							nReserved09;
	UINT8							nField10;
	UINT8							nReserved11[35];
} TCLAPPBLK, *LPTCLAPPBLK;

typedef struct tagHEADER {
	UINT16							nReserved00[2];
	UINT8							nLength;
	UINT16							nUser;
	UINT8							nType;
	UINT8							nPriority;
	UINT8							nReserved09[3];
	UINT8							nSubSystem;
	UINT8							nOpCode;
	UINT16							nResponse;
	UINT16							nValidLen;
	UINT8							nReserved18;
	UINT16							nPayloadLen;
	UINT16							nPayloadOfs;
	UINT8							nReserved23[11];
} HEADER, *LPHEADER;

#define FULLHEADERSIZE				(sizeof(HEADER) + sizeof(FLCAPPBLK))

typedef struct tagS7ONLINEBLOCK {
	HEADER							Hdr;
	union {
		FLCAPPBLK					FlcApp;
		TCLAPPBLK					TclApp;
	};
	UINT8							nPayload[MAX_TCLPAYLOADSIZE];
} S7ONLINEBLOCK, *LPS7ONLINEBLOCK;

typedef struct tagTCLCONSTRUCT {
	UINT8							nField00;						// 0x00
	UINT8							nField01;						// 0x02
	UINT8							nField02;						// 0x01
	UINT8							nField03;						// 0x00
	UINT8							nField04;						// 0x0C
	UINT8							nField05;						// 0x01
	UINT8							nField06;						// 0x00
	UINT8							nField07;						// 0x00
	UINT8							nField08;						// 0x00
	UINT8							nAddr;							// MPI/Profibus address
	UINT8							nField10;						// 0x00
	UINT8							nField11;						// 0x00
	UINT8							nField12;						// 0x00
	UINT8							nField13;						// 0x00
	UINT8							nField14;						// 0x00
	UINT8							nField15;						// 0x01
	UINT8							nField16;						// 0x00
	UINT8							nField17;						// 0x02
	UINT8							nField18;						// 0x01
} TCLCONSTRUCT, *LPTCLCONSTRUCT;

#pragma pack(pop)

typedef INT32 (__stdcall *SCP_OPENFUNC)(LPSTR lpDevice);
typedef INT32 (__stdcall *SCP_CLOSEFUNC)(INT32 nSCP);
typedef INT32 (__stdcall *SCP_SENDFUNC)(INT32 nSCP, UINT16 nLength, LPVOID lpBuf);
typedef INT32 (__stdcall *SCP_RECEIVEFUNC)(INT32 nSCP, UINT16 nTimeOut, LPUINT16 lpDataLen, UINT16 nLength, LPVOID lpBuf);
typedef INT32 (__stdcall *SCP_SINECHWNDFUNC)(INT32 nSCP, HWND hWnd);
typedef INT32 (__stdcall *SCP_GETERRNOFUNC)();

static SCP_OPENFUNC					SCP_Open;
static SCP_CLOSEFUNC				SCP_Close;
static SCP_SENDFUNC					SCP_Send;
static SCP_RECEIVEFUNC				SCP_Receive;
static SCP_GETERRNOFUNC				SCP_GetErrNo;
static HMODULE						m_hLib;

#define IF_ONLINE					"/S7online"
#define IF_CP_L2_1					"/CP_L2_1:"
#define IF_CP_L2_2					"/CP_L2_2:"

static LPSTR						m_pAccessPoint[MAX_IFMODE]	= {NULL, IF_ONLINE, IF_CP_L2_1, IF_CP_L2_2};

// **********************************************************************************
static void __cdecl pCloseLib() {

	FreeLibrary(m_hLib);
	m_hLib = NULL;
	SCP_Open = NULL;
	SCP_Close = NULL;
	SCP_Send = NULL;
	SCP_Receive = NULL;
	SCP_GetErrNo = NULL;
}
// **********************************************************************************
INT32 __stdcall pS7Online_Init(LPS7INTERFACE lpIF, LPINT32 lpSysErr) {
INT32	nRes, nErr;

	nRes = nErr = 0;
	if (!m_hLib) {
		m_hLib = LoadLibrary("S7onlinx.dll");
		if (m_hLib) {
			SCP_Open = (SCP_OPENFUNC) GetProcAddress(m_hLib, "SCP_open");
			SCP_Close = (SCP_CLOSEFUNC) GetProcAddress(m_hLib, "SCP_close");
			SCP_Send = (SCP_SENDFUNC) GetProcAddress(m_hLib, "SCP_send");
			SCP_Receive = (SCP_RECEIVEFUNC) GetProcAddress(m_hLib, "SCP_receive");
			SCP_GetErrNo = (SCP_GETERRNOFUNC) GetProcAddress(m_hLib, "SCP_get_errno");
			if (!SCP_Open || !SCP_Close || !SCP_Send || !SCP_Receive || !SCP_GetErrNo) {
				pCloseLib();
				nRes = S7ERR_LIBERROR;
			}
		}
		else {
			nRes = S7ERR_LIBCANTOPEN;
		}
	}
	if (!nRes) {
		lpIF->nOnlineID = SCP_Open(m_pAccessPoint[lpIF->nMode]);
		if (lpIF->nOnlineID < 0) {
			nRes = S7ERR_SCPERROR;
			nErr = SCP_GetErrNo();
		}
	}
	*lpSysErr = nErr;
	return nRes;
};
// **********************************************************************************
void __stdcall pS7Online_Done(LPS7INTERFACE lpIF, BOOL bCloseLib) {

	if (m_hLib) {
		if (lpIF->nOnlineID >= 0) SCP_Close(lpIF->nOnlineID);
		if (bCloseLib) pCloseLib();
	}
}
// **********************************************************************************
static INT32 __stdcall pInitBuffer(LPS7INTERFACE lpIF, LPVOID lpBuffer, UINT32 nSubSys, UINT32 nOpCode, UINT32 nLen, UINT32 nReference) {
LPS7ONLINEBLOCK		pHdr;
INT32				nRes;

	nRes = pS7_TestInterface(lpIF);
	if (!nRes) {
		pHdr = lpBuffer;
		MemClr(pHdr, FULLHEADERSIZE);
		pHdr->Hdr.nPayloadOfs = pHdr->Hdr.nLength = FULLHEADERSIZE;
		pHdr->Hdr.nUser = lpIF->nUser++;
		pHdr->Hdr.nType = 2;
		pHdr->Hdr.nPriority = PRIORITY_LOW;
		pHdr->Hdr.nSubSystem = nSubSys;
		pHdr->Hdr.nOpCode = nOpCode;
		pHdr->Hdr.nResponse = REQ_WAIT;
		pHdr->Hdr.nValidLen = pHdr->Hdr.nPayloadLen = nLen;
		if (nSubSys == FLC_SUBSYS) {
			pHdr->FlcApp.nOpCode = COM_REQUEST;
			pHdr->FlcApp.nSubSystem = FLC_SUBSYS;
		}
		else {
			pHdr->TclApp.nReference = nReference;
		}
	}
	return nRes;
}
// **********************************************************************************
static INT32 __stdcall pRequest(LPS7INTERFACE lpIF, UINT32 nTimeOut, LPVOID lpSndBuf, LPVOID lpRcvBuf, LPINT32 lpSysErr) {
LPS7ONLINEBLOCK		pSnd, pRcv;
INT32				nRes, nErr, nRetry;
UINT32				nStrt, nAct;
BOOL				bWait;
INT16				nDataLen;

	pSnd = (LPS7ONLINEBLOCK) lpSndBuf;
	pRcv = (LPS7ONLINEBLOCK) lpRcvBuf;
	MemClr(pRcv, sizeof(S7ONLINEBLOCK));
	pRcv->Hdr.nResponse = REQ_WAIT;
	nRetry = 0;
	do {
		nRes = SCP_Send(lpIF->nOnlineID, pSnd->Hdr.nPayloadLen + FULLHEADERSIZE, pSnd);
		if (nRes) {
			nErr = SCP_GetErrNo();
			nRes = S7ERR_SCPERROR;
		}
		else {
			nErr = 0;
		}
		nRetry++;
	} while (nRes && (nRetry < 3));
	if (nRes == S7ERR_NONE) {
		nDataLen = 0;
		nStrt = GetTickCount();
		bWait = TRUE;
		do {
			nRes = SCP_Receive(lpIF->nOnlineID, 0, &nDataLen, sizeof(S7ONLINEBLOCK), pRcv);
			if (nRes) {
				nErr = SCP_GetErrNo();
				nRes = S7ERR_SCPERROR;
			}
			if (nRes || nDataLen) {
				bWait = FALSE;
			}
			else {
				Sleep(0);
				nAct = GetTickCount();
				if ((nAct - nStrt) >= nTimeOut) {
					bWait = FALSE;
					nRes = S7ERR_TIMEOUT;
				}
			}
		} while (bWait);
		if (nRes == S7ERR_NONE) {
			if (nDataLen < FULLHEADERSIZE) {
				nRes = S7ERR_INVALIDRESPONSE;
			}
			else {
				if (pSnd->Hdr.nSubSystem == FLC_SUBSYS) {
					if ((pRcv->Hdr.nResponse)) nRes = S7ERR_REJECTEDRESPONSE;
				}
				else {
					if (((pSnd->Hdr.nOpCode == RECEIVE_DATA) && (pRcv->Hdr.nResponse != OK_EOM_RESP)) || ((pSnd->Hdr.nOpCode != RECEIVE_DATA) && (pRcv->Hdr.nResponse != OK_RESP))) {
						nRes = S7ERR_REJECTEDRESPONSE;
					}
				}
				if (nRes) nErr = pRcv->Hdr.nResponse;
			}
		}
	}
	if (lpSysErr) *lpSysErr = nErr;
	return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7Online_Connect(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
LPS7ONLINEBLOCK		pSnd, pRcv;
LPTCLCONSTRUCT		pCon;
INT32				nRes;

	lpCon->pSndBuf = GetMem(GMM_DEFAULT, 2 * sizeof(S7ONLINEBLOCK));
	if (!lpCon->pSndBuf) return S7ERR_NOMEMORY;
	lpCon->pRcvBuf = lpCon->pSndBuf + sizeof(S7ONLINEBLOCK);
	lpCon->SndPdu.pData = lpCon->pSndBuf + FULLHEADERSIZE;
	lpCon->RcvPdu.pData = lpCon->pRcvBuf + FULLHEADERSIZE;
	pSnd = (LPS7ONLINEBLOCK) lpCon->pSndBuf;
	pRcv = (LPS7ONLINEBLOCK) lpCon->pRcvBuf;

	nRes = pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, OPEN_REQ, 0, 0);
	if (nRes) return nRes;
	nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
	if (nRes) return nRes;
	lpCon->bOpened = TRUE;
	lpCon->nReference = pRcv->TclApp.nReference;

	pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, SEND_CONN_REQ, 126, lpCon->nReference);
	pSnd->TclApp.nField08 = 2;
	pSnd->TclApp.nField10 = 114;
	pCon = (LPTCLCONSTRUCT) pSnd->nPayload;
	MemClr(pCon, MAX_TCLPAYLOADSIZE);
	pCon->nField01 = 0x02;
	pCon->nField02 = 0x01;
	pCon->nField04 = 0x0C;
	pCon->nField05 = 0x01;
	pCon->nAddr = lpCon->nAddr;
	pCon->nField15 = 0x01;
	pCon->nField17 = 0x02;
	pCon->nField18 = 0x01;
	nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
	if (nRes) return nRes;

	pS7PDU_SndInit(lpCon, PDUT_REQUEST, PDUFN_NEGOTIATE, 0, FALSE);
	pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, SEND_EOM_DATA, lpCon->SndPdu.nTotLen, lpCon->nReference);
	nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
	if (nRes) return nRes;

	pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, RECEIVE_DATA, MAX_TCLPAYLOADSIZE, lpCon->nReference);
	nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
	if (nRes) return nRes;
    nRes = pS7PDU_RcvInit(lpCon, lpSysErr);
	if (nRes) return nRes;
	nRes = pS7PDU_SetPDUSize(lpCon);
    return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7Online_Disconnect(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
LPS7ONLINEBLOCK		pSnd, pRcv;
INT32				nRes;

	if (lpCon->bOpened) {
		pSnd = (LPS7ONLINEBLOCK) lpCon->pSndBuf;
		pRcv = (LPS7ONLINEBLOCK) lpCon->pRcvBuf;
		pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, CLOSE_REQ, 0, lpCon->nReference);
		nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
		if (nRes == S7ERR_NONE) {
			lpCon->bOpened = FALSE;
			lpCon->nReference = 0;
		}
	}
	else {
		nRes = 0;
	}
	return nRes;
}
// **********************************************************************************
INT32 __stdcall pS7Online_Exchange(LPS7CONNECTION lpCon, LPINT32 lpSysErr) {
LPS7ONLINEBLOCK		pSnd, pRcv;
INT32				nRes;

	pSnd = (LPS7ONLINEBLOCK) lpCon->pSndBuf;
	pRcv = (LPS7ONLINEBLOCK) lpCon->pRcvBuf;
	pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, SEND_EOM_DATA, lpCon->SndPdu.nTotLen, lpCon->nReference);
	nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
	if (nRes == S7ERR_NONE) {
		pInitBuffer(lpCon->pIF, pSnd, TCL_SUBSYS, RECEIVE_DATA, lpCon->nPDUSize, lpCon->nReference);
		nRes = pRequest(lpCon->pIF, lpCon->nTimeOut, pSnd, pRcv, lpSysErr);
	}
	return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_OnLine_Partners(HANDLE hIF, INT32 nTimeOut, INT32 nSize, LPSTR lpBuf, LPINT32 lpCount) {
LPTIDDATA		pTid;
LPS7INTERFACE	pIF;
S7ONLINEBLOCK	SndBlk;
S7ONLINEBLOCK	RcvBlk;
INT32			nRes, nCount, nErr;

	pTid = _getptd();
	pIF = (LPS7INTERFACE) hIF;
	nErr = 0;
	MemClr(lpBuf, nSize);
	nRes = pInitBuffer(pIF, &SndBlk, FLC_SUBSYS, 0, 128, 0);
	nCount = 0;
	if (nRes == S7ERR_NONE) {
		SndBlk.FlcApp.nServiceCode = FDL_LIFE_LIST_CREATE_LOCAL;
		if (nTimeOut < 1000) nTimeOut = 1000;
		nRes = pRequest(pIF, nTimeOut, &SndBlk, &RcvBlk, &nErr);
		if (nRes == S7ERR_NONE) {
			if (RcvBlk.Hdr.nValidLen < nSize) {
				nCount = RcvBlk.Hdr.nValidLen;
			}
			MemCpy(lpBuf, RcvBlk.nPayload, nSize);
		}
	}
	if (lpCount) *lpCount = nCount;
	pTid->nSysErrNo = nErr;
	pTid->nErrNo = nRes;
    return nRes;
}
// **********************************************************************************
INT32 __stdcall S7_OnLine_BusPars(HANDLE hIF, INT32 nTimeOut, LPBUSPARBLK lpBus) {
LPTIDDATA		pTid;
LPS7INTERFACE	pIF;
S7ONLINEBLOCK	SndBlk;
S7ONLINEBLOCK	RcvBlk;
INT32			nRes, nErr;

	pTid = _getptd();
	pIF = (LPS7INTERFACE) hIF;
	nErr = 0;
	nRes = pInitBuffer(pIF, &SndBlk, FLC_SUBSYS, 0, 256, 0);
	if (nRes == S7ERR_NONE) {
		SndBlk.FlcApp.nServiceCode = FDL_READ_VALUE;
		if (nTimeOut < 1000) nTimeOut = 1000;
		nRes = pRequest(pIF, nTimeOut, &SndBlk, &RcvBlk, &nErr);
		if (nRes == S7ERR_NONE) {
			if (RcvBlk.Hdr.nValidLen == sizeof(BUSPARBLK)) {
				MemCpy(lpBus, RcvBlk.nPayload, sizeof(BUSPARBLK));
			}
			else {
				nRes = S7ERR_INVALIDRESPONSE;
			}
		}
	}
	pTid->nSysErrNo = nErr;
	pTid->nErrNo = nRes;
	return nRes;
}
